/*  $Id: BlastWebServiceClient.java,v 1.2 2008/09/17 22:45:19 camacho Exp $
 * ===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 */

package gov.nih.nlm.ncbi;

import java.io.*;
import java.math.BigInteger;
import java.rmi.RemoteException;

import gov.nih.nlm.ncbi.www.*;
import gov.nih.nlm.ncbi.www.Blast4RequestDocument.*;
import gov.nih.nlm.ncbi.www.Blast4QueueSearchRequestDocument.*;
import gov.nih.nlm.ncbi.www.Blast4QueueSearchRequestLiteDocument.*;
import gov.nih.nlm.ncbi.www.Blast4GetSequencesRequestDocument.*;

/** Main class for the demo Java client to access the NCBI BLAST web service.
 *
 * @author Christiam Camacho
 */
public class BlastWebServiceClient {

	/** Handle to the BLAST web service */
	private static BlastWebServiceStub m_WebService;
	/** Identifier for requests to the web service */
	private final static String kClientId =
		"Demo SOAP Java client developed by NCBI";
	
	/** Main function
	 * @param args command line arguments
	 */
	public static void main(String[] args) {
		try {
			CmdLineParser opts = new CmdLineParser(args);
			
			m_WebService = new BlastWebServiceStub();
			
			if (opts.TestGetMatrices()) {
				GetSupportedMatrices();
			}
			
			if (opts.TestGetPrograms()) {
				GetSupportedPrograms();
			}
			
			if (opts.TestGetTasks()) {
				GetSupportedTasks();
			}
			
			if (opts.TestGetOptions()) {
				GetSupportedOptions();
			}
			
			if (opts.TestGetDatabases()) {
				GetDatabases(opts.GetFileName());
			}
			
			if (opts.TestGetSequences()) {
				GetSequences();
			}
				
			if (opts.TestSubmitLite()) {
				SubmitSearchLite();
			}
			
			if (opts.TestSubmitSearch()) {
				SubmitSearch();
			}
			
			if (opts.TestCheckStatus()) {
				CheckSearchStatus(opts.GetRID());
			}
			
			if (opts.TestGetResults()) {
				GetResults(opts.GetRID());
			}
			
			if (opts.TestGetSearchStrategy()) {
				GetSearchStrategy(opts.GetRID());
			}
				
		} catch (RemoteException e) {
			e.printStackTrace();
		}

	}
	
	/** Retrieves the search strategy for a given RID and saves it to the file
	 * search_strategy.xml
	 * @param RID The RID for the search
	 */
	public static void GetSearchStrategy(String RID) throws RemoteException {
		Blast4GetSearchStrategyRequestDocument reqDoc =
			Blast4GetSearchStrategyRequestDocument.Factory.newInstance();
		reqDoc.addNewBlast4GetSearchStrategyRequest().setBlast4GetSearchStrategyRequestRequestId(RID);
		
		Blast4GetSearchStrategyReplyDocument reply =
			m_WebService.GetSearchStrategy(reqDoc);
		
		final String fname = new String("search_strategy.xml");
		try {
			PrintStream output = new PrintStream(new FileOutputStream(fname));
			output.println(reply.xmlText());
			output.close();		
			System.out.println("Search strategy saved in " + fname);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	/** Retrieves the search results for a given RID
	 * @param RID The RID for the search
	 */
	public static void GetResults(String RID) throws RemoteException {
		Blast4GetSearchResultsRequestDocument reqDoc =
			Blast4GetSearchResultsRequestDocument.Factory.newInstance();
		reqDoc.addNewBlast4GetSearchResultsRequest().setBlast4GetSearchResultsRequestRequestId(RID);
		
		Blast4GetSearchResultsReplyDocument reply =
			m_WebService.GetSearchResults(reqDoc);
		
		final String fname = new String("results.xml");
		try {
			PrintStream output = new PrintStream(new FileOutputStream(fname));
			output.println(reply.xmlText());
			output.close();
			System.out.println("Results can be found in " + fname);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	/** Check the status for a given RID
	 * @param RID The search RID
	 */
	public static void CheckSearchStatus(String RID) throws RemoteException {
		Blast4GetSearchStatusRequestDocument reqDoc =
			Blast4GetSearchStatusRequestDocument.Factory.newInstance();
		reqDoc.addNewBlast4GetSearchStatusRequest().setBlast4GetSearchStatusRequestRequestId(RID);
		
		Blast4GetSearchStatusReplyDocument reply = 
			m_WebService.CheckSearchStatus(reqDoc);
		
		System.out.println("Status=" + 
				reply.getBlast4GetSearchStatusReply().getBlast4GetSearchStatusReplyStatus());
	}
	
	/** Submit a hard coded search using the lite submission method
	 */
	public static void SubmitSearchLite() throws RemoteException {
		Blast4QueueSearchRequestLiteDocument reqDoc = 
			Blast4QueueSearchRequestLiteDocument.Factory.newInstance();
		Blast4QueueSearchRequestLite reqLite =
			reqDoc.addNewBlast4QueueSearchRequestLite();
		reqLite.setBlast4QueueSearchRequestLiteQuery("555");
		reqLite.setBlast4QueueSearchRequestLiteDatabaseName("ecoli");
		
		Blast4OptionsLiteDocument.Blast4OptionsLite opts_lite =
			reqLite.addNewBlast4QueueSearchRequestLiteOptions().addNewBlast4OptionsLite();
		opts_lite.setBlast4OptionsLiteTask("blastn");
		
		Blast4QueueSearchReplyDocument reply = 
			m_WebService.SubmitSearchLite(reqDoc);
		
		System.out.println("RID=" + 
			reply.getBlast4QueueSearchReply().getBlast4QueueSearchReplyRequestId());
	}
	
	/** Submit a hard coded search */
	public static void SubmitSearch() throws RemoteException {
		Blast4QueueSearchRequestDocument reqDoc =
			Blast4QueueSearchRequestDocument.Factory.newInstance();
		Blast4QueueSearchRequest request =
			reqDoc.addNewBlast4QueueSearchRequest();
		request.setBlast4QueueSearchRequestProgram("blastn");
		request.setBlast4QueueSearchRequestService("plain");
		
		// Prepare the query (as seqlocs)
		Blast4QueriesDocument.Blast4Queries.Blast4QueriesSeqLocList seqlocs = 
			request.addNewBlast4QueueSearchRequestQueries().addNewBlast4Queries().addNewBlast4QueriesSeqLocList();
		SeqLocDocument.SeqLoc seqloc = seqlocs.addNewSeqLoc();
		SeqIdDocument.SeqId seqid = seqloc.addNewSeqLocWhole().addNewSeqId();
		seqid.setSeqIdGi(new BigInteger("555"));
		
		// Prepare the subject
		request.addNewBlast4QueueSearchRequestSubject().addNewBlast4Subject().setBlast4SubjectDatabase("ecoli");
		
		Blast4QueueSearchReplyDocument reply = 
			m_WebService.SubmitSearch(reqDoc);		
		System.out.println("RID=" + 
				reply.getBlast4QueueSearchReply().getBlast4QueueSearchReplyRequestId());
	}
	
	/** Retrieve a couple of hard coded sequences identified by gis */
	public static void GetSequences() throws RemoteException {
		
		Blast4GetSequencesRequestDocument reqDoc =
			Blast4GetSequencesRequestDocument.Factory.newInstance();
		Blast4GetSequencesRequest 
			blast4GetSequencesRequest = reqDoc.addNewBlast4GetSequencesRequest();
		
		// Select the database
		Blast4DatabaseDocument.Blast4Database blast4Database =
			blast4GetSequencesRequest.addNewBlast4GetSequencesRequestDatabase().addNewBlast4Database();
		blast4Database.setBlast4DatabaseName("nt");
		blast4Database.addNewBlast4DatabaseType().addNewBlast4ResidueType().
		setValue(Blast4ResidueTypeDocument.Blast4ResidueType.Value.NUCLEOTIDE);

		// Select the sequence data
		Blast4GetSequencesRequest.Blast4GetSequencesRequestSeqIds seqids = 
			blast4GetSequencesRequest.addNewBlast4GetSequencesRequestSeqIds();
		seqids.addNewSeqId().setSeqIdGi(new BigInteger("555"));
		seqids.addNewSeqId().setSeqIdGi(new BigInteger("556"));
			
		Blast4GetSequencesReplyDocument reply = 
			m_WebService.GetSequences(reqDoc);
		
		final String fname = new String("sequences.xml");
		try {
			PrintStream output = new PrintStream(new FileOutputStream(fname));
			output.print(reply.toString());
			output.close();
			System.out.println("Sequences can be found in " + fname);
		} catch (Exception e) {
			e.printStackTrace();
		}				
	}
	
	/** Retrieve databases available for BLAST searches */
	public static void GetDatabases(String fname) throws RemoteException {
		Blast4RequestDocument reqDoc = 
			Blast4RequestDocument.Factory.newInstance();
		Blast4Request blast4Request = reqDoc.addNewBlast4Request();
		blast4Request.setBlast4RequestIdent(kClientId);
		blast4Request.addNewBlast4RequestBody().addNewBlast4RequestBody().addNewBlast4RequestBodyGetDatabases();
		
		Blast4GetDatabasesReplyDocument reply = 
			m_WebService.GetDatabases(reqDoc);
				
		try {
			PrintStream output = new PrintStream(new FileOutputStream(fname));
			output.print(reply.toString());
			output.close();
			System.out.println("Databases are saved in " + fname);
		} catch (Exception e) {
			e.printStackTrace();
		}		
	}

	/** Retrieve matrices supported by BLAST */
	public static void GetSupportedMatrices() throws RemoteException {
		Blast4RequestDocument reqDoc = 
			Blast4RequestDocument.Factory.newInstance();
		Blast4Request blast4Request = reqDoc.addNewBlast4Request();
		blast4Request.setBlast4RequestIdent(kClientId);
		blast4Request.addNewBlast4RequestBody().addNewBlast4RequestBody().addNewBlast4RequestBodyGetMatrices();

		Blast4GetMatricesReplyDocument reply =
			m_WebService.GetSupportedMatrices(reqDoc);
						
		Blast4MatrixIdDocument.Blast4MatrixId[] matrixIds =
			reply.getBlast4GetMatricesReply().getBlast4MatrixIdArray();
		
		System.out.println("Supported BLAST matrices:");
		for (int i = 0; i < matrixIds.length; i++) {
			Blast4MatrixIdDocument.Blast4MatrixId mId = matrixIds[i];
			System.out.println(i+1 + ". " + mId.getBlast4MatrixIdName() + " (" +
				mId.getBlast4MatrixIdResidueType().getBlast4ResidueType().getValue() + ")");
		}	
	}
	
	/** Retrieve tasks supported by BLAST */
	public static void GetSupportedTasks() throws RemoteException {
		Blast4RequestDocument reqDoc = 
			Blast4RequestDocument.Factory.newInstance();
		Blast4Request blast4Request = reqDoc.addNewBlast4Request();
		blast4Request.setBlast4RequestIdent(kClientId);
		blast4Request.addNewBlast4RequestBody().addNewBlast4RequestBody().addNewBlast4RequestBodyGetParamsets();

		Blast4GetParamsetsReplyDocument reply =
			m_WebService.GetSupportedTasks(reqDoc);
						
		Blast4TaskInfoDocument.Blast4TaskInfo[] taskInfoArray =
			reply.getBlast4GetParamsetsReply().getBlast4TaskInfoArray();
		
		System.out.println("Supported BLAST tasks:");
		for (int i = 0; i < taskInfoArray.length; i++) {
			Blast4TaskInfoDocument.Blast4TaskInfo info = taskInfoArray[i];
			System.out.println(i+1 + ". " + info.getBlast4TaskInfoName() + ": " +
				info.getBlast4TaskInfoDocumentation());
		}	
	}
	
	/** Retrieve options supported by BLAST */
	public static void GetSupportedOptions() throws RemoteException {
		Blast4RequestDocument reqDoc =
			Blast4RequestDocument.Factory.newInstance();
		Blast4Request blast4Request = reqDoc.addNewBlast4Request();
		blast4Request.setBlast4RequestIdent(kClientId);
		blast4Request.addNewBlast4RequestBody().addNewBlast4RequestBody().addNewBlast4RequestBodyGetParameters();
		
		Blast4GetParametersReplyDocument reply =
			m_WebService.GetSupportedOptions(reqDoc);
		
		Blast4ParameterInfoDocument.Blast4ParameterInfo[] options =
			reply.getBlast4GetParametersReply().getBlast4ParameterInfoArray();
		
		System.out.println("Supported BLAST options:");
		for (int i = 0; i < options.length; i++) {
			Blast4ParameterInfoDocument.Blast4ParameterInfo info = options[i];
			System.out.println(i+1 + ". " + info.getBlast4ParameterInfoName() + " (" + 
					info.getBlast4ParameterInfoType() + ")");
		}
	}
	
	/** Retrieve programs/services supported by BLAST */ 
	public static void GetSupportedPrograms() throws RemoteException {
		Blast4RequestDocument reqDoc =
			Blast4RequestDocument.Factory.newInstance();
		Blast4Request blast4Request = reqDoc.addNewBlast4Request();
		blast4Request.setBlast4RequestIdent(kClientId);
		blast4Request.addNewBlast4RequestBody().addNewBlast4RequestBody().addNewBlast4RequestBodyGetPrograms();
		
		Blast4GetProgramsReplyDocument reply =
			m_WebService.GetSupportedPrograms(reqDoc);
		
		Blast4ProgramInfoDocument.Blast4ProgramInfo[] programs =
			reply.getBlast4GetProgramsReply().getBlast4ProgramInfoArray();
		
		System.out.println("Supported BLAST programs:");
		for (int i = 0; i < programs.length; i++) {
			Blast4ProgramInfoDocument.Blast4ProgramInfo info = programs[i];
			System.out.println("Program: " + info.getBlast4ProgramInfoProgram());
			String[] services =
                info.getBlast4ProgramInfoServices().getBlast4ProgramInfoServicesEArray();
			System.out.print("Services: " + services[0]);
			for (int j = 1; j < services.length; j++) 
				System.out.print(", " + services[j]);
			System.out.println();
		}
	}
}
