=head1 NAME

gencode_tracking_system : resolve_flags.pl

=head1 DESCRIPTION

Helper script for the GENCODE tracking system.

Resolve defined flags with the solution given.

=head1 SYNOPSIS

  perl resolve_flags.pl
                       -issue-id 300918
                       -flagname splice_GT_AG_none
                       -user felix
                       -solution rejected_splicing_correct
                       -add_notes "non-org support"
                       -env human
  OR

  perl resolve_flags.pl
                       -file filnename.txt
                       -env human

    with the tab-delimited file format:
     user_name  issue_name  flag_name  solution  additional_note


=head1 CONTACT

Felix Kokocinski, fsk@sanger.ac.uk

=head1 COPYRIGHT

Copyright Felix Kokocinski, 2008-2010, 
supported by Wellcome Trust Sanger Institute (UK) 
and National Human Genome Research Institute (USA).

You may distribute this module under the same terms as perl itself, 
citing the original source.

=cut

use strict;
use warnings;
use Getopt::Long;
use gencode_tracking_system::core;
use gencode_tracking_system::config;

my $user_name = "tracking-system";
my $type      = "flag";
my $flag_id;
my $resolved_date = "2010-12-01";
my $solution      = "annotated_new_locus";
my $issue_id      = 0;
my $issue_name    = "";
my $flag_name;
my $add_note;
my $infile;
my $user_id;
my $r = 0;

&GetOptions(
	    'flag-id=s'        => \$flag_id,
	    'issue-id=s'       => \$issue_id,
	    'issue_name=s'     => \$issue_name,
	    'flagname=s'       => \$flag_name,
	    'user=s'           => \$user_name,
	    'solution=s'       => \$solution,
	    'add_notes=s'      => \$add_note,
	    'file=s'           => \$infile,
	    'env=s'            => \$ENVI,
	    'write!'           => \$WRITE,
	   );

gencode_tracking_system::config->set_databases();
print STDERR "Using $DBNAME @ $DBHOST : $DBPORT, $DBNAME ($DBUSER)\n";

#connect to tracking system db
my $tracking_dbh = connect_db($DBHOST, $DBPORT, $DBNAME, $DBUSER, $DBPASS)
  or die "cant connect to to database $DBNAME @ $DBHOST.\n";
if($VERBOSE){ print "Conected to $DBNAME @ $DBHOST.\n" }
my $prepare_hash = prepare_statements($tracking_dbh);

#date of solution
my ($day, $month, $year) = (localtime)[3,4,5];
my $today = sprintf("%04d-%02d-%02d", $year+1900, $month+1, $day);
$resolved_date = $today unless($resolved_date);

#read flag solutions from file
if($infile){

  open(F, "<$infile") or die "Cant read file $infile!\n";
  while(<F>){
    chomp $_;
    #adjust for different file format!
    ($user_name, $issue_name, $flag_name, $solution, $add_note) = split("\t", $_);

    if(!($user_name && $issue_name && $flag_name && $solution)){
      print STDERR "Not enoght information: ".$_."\n";
      next;
    }

    $user_id = get_user_id( $user_name, $tracking_dbh ) or die "Cant find user $user_name!\n";

    resolve($issue_name, $issue_id, $flag_name, $solution, $user_id, $add_note);

    #last if($r>10);
  }
  close(F);
}
#or from command-line options
else{
  $user_id = get_user_id( $user_name, $tracking_dbh );
  resolve($issue_name, $issue_id, $flag_name, $solution, $user_id, $add_note);
}

print STDERR "\nAnalysed $r flags.\n";


sub resolve{
  my ($i_name, $i_id, $f_name, $sol, $u_id, $note) = @_;

  #get id if only transcript name is given
  if($i_name and !$i_id){
    my $search_object = get_data_by_name($tracking_dbh, "transcript", $issue_name);
    die "Cant get transcript $issue_name!\n" if(!$search_object or !($search_object->{'id'}));
    $i_id = $search_object->{'id'};
  }

  $note =~ s/\"//g if($note);
  if(!($sol =~ /^(rejected_|annotated_).+/)){
    warn "Invalid flag solution $sol for $i_id\n";
  }

  #resolve flag and write history
  my $res = resolve_flag($tracking_dbh, $i_id, $f_name, $resolved_date, $sol, $u_id, $note, $prepare_hash);

  if($res ne "RESOLVED"){
    print "Did not resolve $i_id ($flag_name):\t$res\n";
  }
  $r++;
}


__END__

perl resolve_flags.pl -env dev -nowrite -file ~/0_tmp/qc_import/test.txt

