#!/usr/bin/env perl

use strict;
use Data::Dumper;
use Bio::EnsEMBL::Registry;
use Bio::EnsEMBL::Funcgen::DBSQL::DBAdaptor;
use Getopt::Long;
use Hash::Util qw( lock_hash lock_keys );

=head1
=head2 Description

  This script takes the transcript_info_file generated by 
  examine_transcript.pl which
  
    for every transcript shows all probes and probe sets it maps to 
    
  into a map 
    from every probeset to the transcript it matches.
  
  This is only done for probesets. In order to avoid excessively large files
  this has a mandatory array_name parameter which filters the matches to
  probesets from only one array.

=cut

=head1

[mnuhn@ebi-003 ensembl-funcgen]$ grep -c "VAR1 = {" /nfs/nobackup/ensembl/mnuhn/array_mapping/temp/mus_musculus/transcript_info.pl
124168
[mnuhn@ebi-003 ensembl-funcgen]$ grep -c "VAR1 = {" /nfs/nobackup/ensembl/mnuhn/array_mapping/temp/homo_sapiens/transcript_info.pl
215929

bsub -q production-rh7 -M64000 -R"select[mem>64000] rusage[mem=64000]" -Is $SHELL

time compute_probeset_transcript_hits.pl \
  --array_name HuGene-2_0-st-v1 \
  --transcript_info_file  /nfs/nobackup/ensembl/mnuhn/array_mapping/temp/homo_sapiens/transcript_info.pl \
  --probeset_transcript_hits_file /nfs/nobackup/ensembl/mnuhn/array_mapping/temp/homo_sapiens/probeset_transcript_hits_file.HuGene-2_0-st-v1.pl

time compute_probeset_transcript_hits.pl \
  --array_name MG-U74Bv2 \
  --transcript_info_file       /nfs/nobackup/ensembl/mnuhn/array_mapping/temp/mus_musculus/transcript_info.pl \
  --probeset_transcript_hits_file /nfs/nobackup/ensembl/mnuhn/array_mapping/temp/mus_musculus/probeset_transcript_hits_file.pl
  
Uses 28Gb.

=cut

my $transcript_info_file;
my $probeset_transcript_hits_file;
my $array_name;

GetOptions (
   'array_name=s'                 => \$array_name,
   'transcript_info_file=s'       => \$transcript_info_file,
   'probeset_transcript_hits_file=s' => \$probeset_transcript_hits_file,
);

use Bio::EnsEMBL::Utils::Logger;
my $logger = Bio::EnsEMBL::Utils::Logger->new();
$logger->init_log;

use Bio::EnsEMBL::Funcgen::Parsers::DataDumper;
my $parser = Bio::EnsEMBL::Funcgen::Parsers::DataDumper->new;

$logger->info("Reading transcript info " . $transcript_info_file . ".\n");

my $num_transcripts_seen        = 0;
my $print_life_sign_after_every = 1000;

my %object_transcript_hits;

open my $probeset_transcript_hits_file_fh,    '>', $probeset_transcript_hits_file;

$parser->parse({
  data_dumper_file => $transcript_info_file,
  call_back        => \&process_transcript_info,
});

$logger->info("Done reading transcript info\n");
$logger->info("Writing file $probeset_transcript_hits_file\n");

# $probeset_transcript_hits_file_fh->print(Dumper(\%object_transcript_hits));
my $probe_transcript_hits_for_current_array = $object_transcript_hits{$array_name};

my @probe_ids = keys %$probe_transcript_hits_for_current_array;
foreach my $current_probe_id (@probe_ids) {

  my $probe_transcript_hits_for_current_probe = $probe_transcript_hits_for_current_array->{$current_probe_id};

  $probeset_transcript_hits_file_fh->print(
    Dumper(
      {
        $array_name => 
          {
            $current_probe_id => $probe_transcript_hits_for_current_probe
          }
      }
    )
  );
}

$logger->info("Done.\n");
$logger->finish_log;
$logger->info("All done.\n");

sub process_transcript_info {

  my $transcript_info = shift;
  
  $num_transcripts_seen++;
  
  if ($num_transcripts_seen % $print_life_sign_after_every == 0) {
    $logger->info("Seen $num_transcripts_seen transcripts.\n");
  }
  
  my $probe_hits_by_array = $transcript_info->{probe_hits_by_array};
  my @array_names = keys %$probe_hits_by_array;
  
  ARRAY:
  foreach my $current_array_name (@array_names) {
  
    next ARRAY unless($current_array_name eq $array_name);
  
    my $current_probe_hits = $probe_hits_by_array->{$current_array_name};
    my $is_probeset_array = $current_probe_hits->{probeset_array};
    
    if (!$is_probeset_array) {
      next ARRAY;
    }
    
    my $probe_set_hits = $current_probe_hits->{probesets};
    my @probe_set_ids = keys %$probe_set_hits;
    
    foreach my $current_probe_set_id (@probe_set_ids) {
    
      my $transcript_stable_id = $transcript_info->{transcript_stable_id};
      my $current_hits = $probe_set_hits->{$current_probe_set_id};
      
      $object_transcript_hits{$current_array_name}{$current_probe_set_id}{$transcript_stable_id} = $current_hits;
    }
  }
}

