#!/usr/bin/env python3
# See the NOTICE file distributed with this work for additional information
# regarding copyright ownership.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Calculate genomic coverage for a sequence chunk in a HAL file."""

from argparse import ArgumentParser
import json
from pathlib import Path
import re
import subprocess
from tempfile import SpooledTemporaryFile
from typing import Dict, Iterable, TextIO, Union


def hal_genomic_coverage(
    hal_path: Union[Path, str],
    ref_genome: str,
    ref_sequence: str,
    start: int,
    length: int,
    target_genomes: Iterable[str],
    hal_alignment_depth_exe: Union[Path, str] = "halAlignmentDepth",
) -> Dict:
    """Uses halAlignmentDepth to get genomic coverage for the given sequence chunk.

    Args:
        hal_path: Input HAL file.
        ref_genome: Reference genome against which coverage is calculated.
        ref_sequence: Sequence in reference genome for which coverage is calculated.
        start: Start of sequence chunk for which coverage is calculated.
        length: Length of sequence chunk for which coverage is calculated.
        target_genomes: Target genomes to be considered for genomic coverage calculation.
        hal_alignment_depth_exe: Path of halAlignmentDepth executable. By default, assumed
            to be available via the PATH environment variable.

    Returns:
        Dictionary of genomic coverage stats for the
        specified reference genome sequence chunk.
    """
    cmd_args = [
        hal_alignment_depth_exe,
        hal_path,
        ref_genome,
        "--noAncestors",
        "--refSequence",
        ref_sequence,
        "--targetGenomes",
        ",".join(target_genomes),
        "--start",
        str(start),
        "--length",
        str(length),
    ]

    rollover_limit = 10_485_760
    with SpooledTemporaryFile(max_size=rollover_limit, mode="w+t") as tmp_file_obj:
        try:
            subprocess.run(
                cmd_args, stdout=tmp_file_obj, text=True, encoding="ascii", check=True
            )
        except subprocess.CalledProcessError as exc:
            status_type = "exit code" if exc.returncode > 0 else "signal"
            raise RuntimeError(
                f"halAlignmentDepth terminated with {status_type} {abs(exc.returncode)}"
                f" for sequence '{ref_sequence}' of genome 'ref_genome'"
            ) from exc

        tmp_file_obj.flush()
        tmp_file_obj.seek(0)
        result_part = load_genomic_coverage_wiggle(tmp_file_obj)

    return result_part[ref_sequence]


def load_genomic_coverage_wiggle(
    wiggle_file_obj: Union[SpooledTemporaryFile, TextIO]
) -> Dict:
    """Load data from wiggle file generated by halAlignmentDepth.

    Args:
        wiggle_file_obj: Input wiggle file object.

    Returns:
        Nested dictionary of coverage stats, with the key being the sequence name,
        and the value being a dictionary of coverage stats for that sequence.
    """
    declaration_line_re = re.compile(
        r"fixedStep chrom=(?P<chrom>\S+) start=(?P<start>[0-9]+) step=(?P<step>[0-9]+)\s*"
    )

    cov_stats = {}
    curr_seq_name = None
    for line in wiggle_file_obj:
        try:
            aligned_genome_count = int(line)
        except ValueError as exc:
            if match := declaration_line_re.fullmatch(line):
                curr_seq_name = match["chrom"]
                if curr_seq_name in cov_stats:
                    raise ValueError(
                        f"multiple occurrences of sequence '{curr_seq_name}' found"
                    ) from exc
                cov_stats[curr_seq_name] = {
                    "num_aligned_positions": 0,
                    "num_positions": 0,
                    "start": int(match["start"]),
                    "step": int(match["step"]),
                }
            elif line.startswith("variableStep"):
                raise ValueError("variableStep blocks not supported") from exc
            else:
                raise ValueError(f"failed to parse wiggle line: {line}") from exc

        else:
            if aligned_genome_count > 0:
                cov_stats[curr_seq_name]["num_aligned_positions"] += 1
            cov_stats[curr_seq_name]["num_positions"] += 1

    return cov_stats


def main() -> None:
    """Main function of script."""

    parser = ArgumentParser(description=__doc__)
    parser.add_argument("hal_path", help="Input HAL file.")
    parser.add_argument(
        "ref_genome", help="Name of genome for which genomic coverage is calculated."
    )
    parser.add_argument(
        "--ref-sequence",
        metavar="STR",
        help="Name of sequence for which genomic coverage is calculated.",
    )
    parser.add_argument(
        "--start",
        metavar="INT",
        required=True,
        type=int,
        help="Start of sequence chunk (0-based).",
    )
    parser.add_argument(
        "--length",
        metavar="INT",
        required=True,
        type=int,
        help="Length of sequence chunk.",
    )
    parser.add_argument(
        "--target-genomes",
        metavar="STR",
        required=True,
        help="Comma-separated list of target genomes.",
    )

    parser.add_argument(
        "--hal_alignment_depth_exe",
        metavar="STR",
        help="Path of halAlignmentDepth executable. By default, assumed"
        " to be available via the PATH environment variable.",
    )

    args = parser.parse_args()

    hal_cov_result = hal_genomic_coverage(
        args.hal_path,
        args.ref_genome,
        args.ref_sequence,
        args.start,
        args.length,
        args.target_genomes.split(","),
        hal_alignment_depth_exe=args.hal_alignment_depth_exe,
    )
    obs_num_positions = hal_cov_result["num_positions"]

    if obs_num_positions != args.length:
        raise ValueError(
            f"sequence-length mismatch: {obs_num_positions} vs {args.length}"
        )

    output = {
        "num_positions": obs_num_positions,
        "num_aligned_positions": hal_cov_result["num_aligned_positions"],
    }

    print(json.dumps(output))


if __name__ == "__main__":
    main()
