
=head1 NAME

gencode_tracking_system / set_priorities

=head1 DESCRIPTION

Helper script for the GENCODE tracking system, to be run nightly.
Connect to the internal tracking system db, analyze flags and set priorities and active flags.

You can add specific flag types with their priorities here.
You can also set priorities by executing specific queries or combined flags.

=head1 CONTACT

Felix Kokocinski, fsk@sanger.ac.uk

=head1 COPYRIGHT

Copyright Felix Kokocinski, 2008-2010, 
supported by Wellcome Trust Sanger Institute (UK) 
and National Human Genome Research Institute (USA).

You may distribute this module under the same terms as perl itself, 
citing the original source.

=cut

use strict;
use warnings;
use Getopt::Long;

use gencode_tracking_system::core;
use gencode_tracking_system::config;

#Genomic regions annotated to a "finished" degree
my $finished_chroms = '"1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "13", "20", "21", "22", "X", "Y"';

#flag priority definitions
#if not defined explicitely flagged issues are set to priority "normal", un-flagged issues are set to "low"

my %flags = (
	     'other_problem'             => $PRIORITY{'normal'},
	     'novel_locus'               => $PRIORITY{'high'},

	    );

&GetOptions(
	    'verbose!'      => \$VERBOSE,
	    'write!'        => \$WRITE,
	   );

my $priority_id;
my $c = 0;
my $user_name = "trackingsystem";
my %issue_flags;
my ($tracking_dbh, $sql);
my @prepared;

#connect to tracking system db
$tracking_dbh = connect_db($DBHOST, $DBPORT, $DBNAME, $DBUSER, $DBPASS)
  or die "cant connect to to database $DBNAME @ $DBHOST.\n";
print "Conected to $DBNAME @ $DBHOST.\n";

#base source name, add other variables to use more than one core annotation
my ($core_source) = keys %CORE_SERVER;

#general prepares
my $prepare_hash = prepare_statements($tracking_dbh);

#custom prepares for URGENT attention
$sql = "UPDATE issues SET active_flags = ? WHERE id = ?";
my $prepare_active_flags_sth = custom_prepare($tracking_dbh, $sql);

#combination of missing_exon, missing_intron, missing_cdna
$sql = 'SELECT id FROM issues '.
       'WHERE active_flags like "%missing_exon%" AND active_flags like "%missing_intron%" '.
       'AND active_flags like "%missing_cdna%";';
my $prepare_combined_sth = custom_prepare($tracking_dbh, $sql);
push(@prepared, $prepare_combined_sth);

#missing_ccds on Sanger chromosomes and not recently updated
$sql = 'SELECT id FROM issues '.
       'WHERE active_flags like "%missing_ccds%" AND Tchrom in('.$finished_chroms.')';
#my $lastdumpdate = '2010-02-01';
#if($lastdumpdate){
#  $sql .= ' AND updated_on < $lastdumpdate';
#}
$prepare_combined_sth = custom_prepare($tracking_dbh, $sql);
push(@prepared, $prepare_combined_sth);

#missing_ccds in annotated regions on non-Sanger chromosomes
$sql = 'SELECT i.id FROM flags f, issues i '.
       'WHERE f.flag_name="missing_ccds" AND f.checked_date IS NULL AND i.id=f.issue_id '.
       'AND Tchrom NOT IN('.$finished_chroms.') AND description NOT LIKE "%no_havana_genes%";';
$prepare_combined_sth = custom_prepare($tracking_dbh, $sql);
push(@prepared, $prepare_combined_sth);


#get all active flag types
my $all_flags = get_flag_types($tracking_dbh, 1);
#add to priorities if neccessary
foreach my $flag_name (keys %$all_flags){
  if(!defined $flags{$flag_name}){
    $flags{$flag_name} = $PRIORITY{'normal'};
  }
}

#re-set seen-flags, NOT NEEDED?!
#reset_seen_flags($tracking_dbh, 'flags');
reset_seen_flags($tracking_dbh, 'issues');

#system user
my $user_id = get_user_id( $user_name, $tracking_dbh );

#go through all flag names, collect flags & issues
foreach my $flag_name (keys %flags){

  #next unless ($flag_name eq 'congo_nc'); ##############################

  print "Looking for flag $flag_name.";
  #get specific unresolved flags (from flags table)
  my $flags = get_flag($tracking_dbh, undef, $flag_name, undef, undef, 'no');

  if(!scalar @$flags){
    print "\n";
    next;
  }
  else{
    print " Found ".(scalar @$flags)." flags.\n";
  }

  #find all issues with this flag
  foreach my $flag (@$flags){

    #add the flag type to the target issue
    if(!defined $issue_flags{ $flag->{'issue_id'} }){
      $issue_flags{ $flag->{'issue_id'} } = [];

      #set seen-flag for transcript
      #set_seen_flag($prepare_hash, $flag->{'id'}, 'flag', "1");
      set_seen_flag($prepare_hash, $flag->{'issue_id'}, 'issue', "1");
    }

    push(@{$issue_flags{ $flag->{'issue_id'} }}, $flag_name);

    #if($flag->{'issue_id'} == 157084){ print "FLAG ".$flag->{'id'}.", ".$flag->{'flag_name'}."\n"; }
    #if($c++>1){ last } ################################

  }

}

print "\nHave ".(scalar keys %issue_flags)." total issue-flag entries.\n\n";

#store flags as "active_flags" to issues
print "Storing active flags.\n";
if($WRITE){

  foreach my $flag_issue_id (keys %issue_flags){
    my %distinct_flag_hash = ();
    foreach my $flag_name (@{ $issue_flags{ $flag_issue_id } }){
      $distinct_flag_hash{$flag_name} = 1;
    }
    my @distinct_flag_array = keys %distinct_flag_hash;
    print "FLAGS $flag_issue_id: ".(join(", ", @distinct_flag_array))."\n" if($VERBOSE);

    set_issue_flags($tracking_dbh, $flag_issue_id, \@distinct_flag_array, $prepare_active_flags_sth);
  }

}

print "Removing old active_flags.\n";
clear_active_flags($tracking_dbh, 1);

#set combined priorities higher
$c = 0;
print "Raising combined flag issue priorities.\n";
$priority_id = $PRIORITY{'urgent'};
foreach my $combined_sth (@prepared){
  $combined_sth->execute();
  while(my $issue_id = $combined_sth->fetchrow_array){
    if($WRITE){
      set_issue_priority($tracking_dbh, $prepare_hash, $issue_id, $priority_id, $user_id);
    }
    $c++;
    #remove from rest
    delete $issue_flags{$issue_id};
  }
  print "combined: $c.\n";
  $c = 0;
  $combined_sth->finish;
}

#set other issue priority according to the different flag types
print "Setting flag-specific issue priorities.\n";
foreach my $flag_issue_id (keys %issue_flags){

  my $highest_priority = 0;
  foreach my $flag_name ( @{ $issue_flags{ $flag_issue_id } } ){
    if($flags{$flag_name} > $highest_priority){
      $highest_priority = $flags{$flag_name};
    }
  }
#  if($highest_priority == 2){
#    $priority_id = $PRIORITY{'normal'};
#    print "\n  -> Setting ".(scalar @$flags)." priorities to $priority_id (normal)";
#  }
#  elsif($highest_priority == 3){
#    $priority_id = $PRIORITY{'high'};
#    print "\n  -> Setting ".(scalar @$flags)." priorities to $priority_id (high)";
#  }

  set_issue_priority($tracking_dbh, $prepare_hash, $flag_issue_id, $highest_priority, $user_id);
}

#re-set priorities to "low" of issues not "seen"
print "Lowering other issue priorities.\n";
$c = 0;
$priority_id = $PRIORITY{'low'};
my $core_category_id = get_category_id( $CORE_SERVER{$core_source}->{'category'}, $tracking_dbh );
foreach my $unseen_id ( @{ get_unseen_issues($tracking_dbh, undef, $priority_id) } ){
  set_issue_priority($tracking_dbh, $prepare_hash, $unseen_id, $priority_id, $user_id);
  $c++;
}
print "Found $c.\n";


#disconnect tracking system db
$prepare_active_flags_sth->finish;
release_prepares($prepare_hash);
disconnect_db($tracking_dbh);

print "\nDONE\n";


__END__


