=head1 NAME

gencode_tracking_system::core

=head1 DESCRIPTION

Perl API for the tracking system AnnoTrack.

The AnnoTrack system records and organizes annotations of genomic loci.

It was developed for the GENCODE part of the ENCODE project [http://www.genome.gov/10005107], 
an international research effort to decipher the human genome.

The GENCODE project is headed by Tim Hubbard at the Wellcome Trust Sanger Institute, UK.
 [http://www.sanger.ac.uk/gencode]

The AnnoTrack system and the data is created & coordinated by Felix Kokocinski 
at the Wellcome Trust Sanger Institute, UK.

Homepage: http://www.sanger.ac.uk/annotrack

=head1 SYNOPSIS

 exported methods:
	connect_db
	disconnect_db
	connect_das
	connect_ensembl
	get_das_response
	show_das_response
	get_user_id
	get_tracker_id
	get_category_id
        get_secondary_id
	get_entry_points
	prepare_statements
	release_prepares
	store_features
	is_active
	make_active
	compare_dates
	check_parent
        get_project_for_category
	insert_element
	remove_subfeatures
	print_element
	add_category
	name_is_present
	compare_coordinates
	update_element
	update_custom_value
	update_element_description
	set_seen_flag
	reset_seen_flags
	analyse_seen_flags
	get_unseen_issues
        get_status
	change_status
	change_subfeature_status
	get_custom_value
	set_flag
	get_flag
	resolve_flag
	set_issue_flags
	clear_active_flags
	custom_prepare
	write_history
	get_issues
	get_subfeatures
	get_data_by_id
	get_data_by_name
	get_genes
	get_transcripts
	create_gene_object
	set_date
	set_issue_priority
	delete_projects
	send_mail
        issue_relation
        store_tec
        store_tec_feature
        get_flag_types
        create_general_project
        get_core_annotation

 private methods are prefixed with _

=head1 CONTACT

Felix Kokocinski, fsk@sanger.ac.uk

=head1 COPYRIGHT

Copyright Felix Kokocinski & WTSI 2008-20010, 
supported by Wellcome Trust Sanger Institute (UK) 
and National Human Genome Research Institute (USA).

Released under GPL v2.

=head1 METHODS

=cut

package gencode_tracking_system::core;

use strict;
use warnings;
use Bio::Das::Lite;
use MIME::Lite;
use DBI;
use gencode_tracking_system::config;
use Bio::EnsEMBL::DBSQL::DBAdaptor;

use base 'Exporter';

our @EXPORT = qw( connect_db disconnect_db connect_das get_das_response show_das_response insert_element
		  print_element name_is_present prepare_statements store_gene get_user_id get_category_id
		  release_prepares get_data_by_id get_issues set_issue_priority add_category set_seen_flag
		  get_tracker_id compare_coordinates store_features get_data_by_name add_issue_note
		  reset_seen_flags delete_projects analyse_seen_flags set_flag get_flag get_entry_points
		  set_issue_flags send_mail et_genes get_transcripts test_do custom_prepare get_subfeatures
		  get_unseen_issues clear_active_flags connect_ensembl get_custom_value write_history
		  change_project_status change_issue_status remove_subfeatures get_project_for_category
		  get_status issue_relation store_tec store_tec_feature get_flag_types create_general_project
		  get_secondary_id get_core_annotation );

=head2 connect_db

  Arg [1]    : host
  Arg [2]    : port
  Arg [3]    : db name
  Arg [4]    : user name
  Arg [5]    : password
  Description: connect to (tracking system) db DBI-style
  Returntype : DBI connection handle

=cut

sub connect_db {
  my ($dbhost, $dbport, $dbname, $dbuser, $dbpass) = @_;

  my $dsn = "DBI:mysql:$dbname:$dbhost:$dbport";
  my $dbh = DBI->connect($dsn, $dbuser, $dbpass)
    or die "cant connect to database $dbname @ $dbhost.\n";

  return $dbh;
}


=head2 disconnect_db

  Arg [1]    : db handle
  Description: disconnect DBI database connection
  Returntype : none

=cut

sub disconnect_db {
  my ($tracking_dbh) = @_;
  $tracking_dbh->disconnect()
    or die "cant disconnect from database.\n";
}


=head2 connect_das

  Arg [1]    : DNS name
  Arg [2]    : proxy
  Description: connect to das source
  Returntype : Bio::DAS::Lite connection

=cut

sub connect_das {
  my ($dns, $proxy) = @_;

  my $das = Bio::Das::Lite->new({
				 'timeout'    => $DASTIMEOUT,
				 'dsn'        => $dns,
				 'http_proxy' => $proxy,
				});
  if(!$das){
    print "Can not connect to das source.\n";
    return 0;
  }
  elsif($VERBOSE){
    print "Connected to das source.\n";
  }

  return $das;
}


=head2 connect_ensembl

  Arg [1]    : host
  Arg [2]    : port
  Arg [3]    : db name
  Arg [4]    : user name
  Arg [5]    : password
  Description: connect to an ensembl-style database
  Returntype : DB connection

=cut

sub connect_ensembl {
  my ($dbhost, $dbport, $dbname, $dbuser, $dbpass) = @_;

  my $dbc = new Bio::EnsEMBL::DBSQL::DBAdaptor(
					       -host    => $dbhost,
					       -user    => $dbuser,
					       -pass    => $dbpass,
					       -port    => $dbport,
					       -dbname  => $dbname,
					      );

  if(!$dbc){
    print "Can not connect to ENSEMBL db.\n";
    return 0;
  }

  return $dbc;
}


=head2 get_das_response

  Arg [1]    : DAS connection
  Arg [2]    : chromosome
  Arg [3]    : (optional) start
  Arg [4]    : (optional) end
  Arg [5]    : (optional) DAS type
  Description: get response from das server for region
  Returntype : DAS response

=cut

sub get_das_response {
  my ($das, $chrom, $start, $end, $type_request) = @_;

  if(!($das && $chrom)){
    die "Cant get DAS data without source and chromosome.\n";
  }

  my $region = '';
  if($start and $end){
    $region = ":".$start.",".$end;
  }

  #fetch DAS features
  my $response = $das->features({
				 'segment' => $chrom.$region,
				 'type'    => $type_request,
				});
  if(!$response){
    print "Can not get a response for ".$chrom.$region.".\n";
  }

  return $response;
}


=head2 show_das_response

  Arg [1]    : DAS response
  Description: print all elements found in a DAS response to STDOUT for debugging
  Returntype : none

=cut

sub show_das_response {
  my ($response) = @_;

  while (my ($url, $features) = each %$response) {
    print $url."\n\n";

    if(!$features){
      print "Empty response.\n";
      exit 1;
    }
    if(ref($features) eq "ARRAY"){
      foreach my $feature (@$features) {
	print "\nFEATURE\n";
	foreach my $k (keys %{$feature}){
	  if($k eq "group"){
	    print "group:\n";
	    my $grouphash = $feature->{$k}->[0];
	    foreach my $l (keys %{$grouphash}){
	      print $l.": ".$grouphash->{$l}."\n";
	    }
	  }
	  elsif(ref($feature->{$k}) eq "ARRAY"){
	    print print $k.":\n";
	    foreach my $m (@{ $feature->{$k} }){
	      print "   ".$m."\n";
	    }
	  }
	  else{
	    print $k.": ".$feature->{$k}."\n";
	  }
	}
      }
    }
    else{
      foreach my $l (keys %{$features}){
	print "H-".$l.": ".$features->{$l}."\n";
      }
    }
  }
}


=head2 get_user_id

  Arg [1]    : user name
  Arg [2]    : db handle
  Description: get the user id using the login name
  Returntype : int user id

=cut

sub get_user_id {
  my ($user_name, $dbh) = @_;
	
  my $user_id = 0;
  my $sth = $dbh->prepare( "SELECT id from users where login=?" );
  $sth->execute($user_name) or die "Cant execute query to get user-id.\n";
  ($user_id) = $sth->fetchrow_array();
  $sth->finish;

  return $user_id;
}


=head2 get_tracker_id

  Arg [1]    : tracker name
  Arg [2]    : db handle
  Description: get the id of the tracker used
  Returntype : int tracker id

=cut

sub get_tracker_id {
  my ($tracker_name, $dbh) = @_;
	
  my $tracker_id = 0;
  my $sth = $dbh->prepare( "SELECT id from trackers where name=?" );
  $sth->execute($tracker_name) or die "Cant execute query to get tracker-id.\n";
  ($tracker_id) = $sth->fetchrow_array();
  $sth->finish;

  return $tracker_id;
}


=head2 get_category_id

  Arg [1]    : category name
  Arg [2]    : db handle
  Description: get the category id using the category name
  Returntype : int, category id

=cut

sub get_category_id {
  my ($category, $dbh) = @_;
	
  my $category_id = 0;
  my $sth = $dbh->prepare( "SELECT id from issue_categories where name=?" );
  $sth->execute($category) or die "Cant execute query to get category-id.\n";
  ($category_id) = $sth->fetchrow_array();
  $sth->finish;

  return $category_id;
}


=head2 get_secondary_id

  Arg[1]     : primary id
  Arg [2]    : db handle
  Description: get a seoncdary id connected with the given id,
               example: ENST00000239223 -> OTTHUMT00000252943
  Returntype : int, secondary id

=cut

sub get_secondary_id{
  my ($transcript_id, $prepare_hash) =@_;

  my $secondary_id = "";
  $prepare_hash->{'id_conversion_1'}->execute($transcript_id);
  ($secondary_id) = $prepare_hash->{'id_conversion_1'}->fetchrow_array();

  if(!$secondary_id){
    $prepare_hash->{'id_conversion_2'}->execute($transcript_id);
    ($secondary_id) = $prepare_hash->{'id_conversion_2'}->fetchrow_array();
  }

  return $secondary_id;
}


=head2 get_entry_points

  Arg [1]    : DAS connection handle
  Arg [2]    : bool, get static version (if DAS source is down)
  Description: fetch entry points from DAS server:
               for the otter-das server this return all chromosomes and their lengths
  Returntype : hash ref with chromosome-name => sequence-length

=cut

sub get_entry_points {
  my ($das, $static) = @_;

  my %chrom_lens;

  if($static){
    #use this to re-create static version:
    %chrom_lens = (
		   '11' => '134452384',
		   '21' => '46944323',
		   '7'  => '158821424',
		   'Y'  => '57772954',
		   '2'  => '242951149',
		   '17' => '78774742',
		   '22' => '49691432',
		   '1'  => '247249719',
		   '18' => '76117153',
		   '13' => '114142980',
		   '16' => '88827254',
		   '6'  => '170899992',
		   'X'  => '154913754',
		   '3'  => '199501827',
		   '9'  => '140273252',
		   '12' => '132349534',
		   '14' => '106368585',
		   '15' => '100338915',
		   '20' => '62435964',
		   '8'  => '146274826',
		   '4'  => '191273063',
		   '10' => '135374737',
		   '19' => '63811651',
		   '5'  => '180857866'
		  );
  }
  else{

    my $entry_points = $das->entry_points();

    if(!$entry_points){
      warn "\nCould not get entry points!\n\n";
    }
    else{
      foreach my $k (keys %$entry_points){
	foreach my $l (@{$entry_points->{$k}}){
	  foreach my $segment (@{ $l->{"segment"} }){
	    $chrom_lens{ $segment->{"segment_id"} } = $segment->{"segment_size"};
	  }
	}
      }
    }

  }

  return \%chrom_lens;
}


=head2 prepare_statements

  Arg [1]    : db handle
  Description: prepare a set of db statement handles for standard functions for
               accessing & storing objects (genes, transcripts and subfeatures)
  Returntype : hash ref with prepared db handles

=cut

sub prepare_statements {
  my ($dbh) = @_;

  my %prepare_hash = ();
  my ($sql, $sth);

  #for genes
  $sql = "INSERT INTO projects SET name=?, description=?, created_on=?, updated_on=?, ".
         "identifier=?, Gchrom=?, Gstart=?, Gend=?, Gstrand=?, is_public=1;";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'gene_1'} = $sth;

  $sql = "INSERT INTO projects_trackers SET project_id=?, tracker_id=?;";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'gene_2'} = $sth;

  $sql = "INSERT INTO custom_values SET customized_type=?, customized_id=?, custom_field_id=?, value=?;";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'gene_3'} = $sth;

  $sql = "INSERT INTO members SET user_id=?, role_id=?, created_on=now(), project_id=?;";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'gene_4'} = $sth;

  $sql = "INSERT INTO enabled_modules set project_id=?, name=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'gene_5'} = $sth;

  #categories
  #$sql = "INSERT INTO issue_categories SET project_id=?, name=?;";
  #my $dbh_gene_6 = $dbh->prepare($sql) or die $dbh->errstr;
  #$prepare_hash{'gene_6'} = $dbh_gene_6;

  #for transcripts
  $sql = "INSERT INTO issues SET tracker_id=?, project_id=?, subject=?, description=?, category_id=?, ".
	 "status_id=?, priority_id=?, author_id=?, created_on=?, updated_on=?, ".
         "start_date=now(), Tchrom=?, Tstart=?, Tend=?, Tstrand=?;";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'transcript_1'} = $sth;
  #re-use custom value prepare
  $prepare_hash{'transcript_3'} = $prepare_hash{'gene_3'};

  #for tecs
  $sql = "INSERT INTO tecs SET issue_id=?, issue_version=?, issue_name=?, ".
         "flag_id=?, sel_remark=?, user=?, result=?;";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'new_tec'} = $sth;

  #for tec-features
  $sql = "INSERT INTO tec_features SET tec_id=?, ftype=?, chromosome=?, ".
         "start=?, end=?, strand=?, sequence=?;";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'new_tecfeature'} = $sth;

  #for subfeatures
  $sql = "INSERT INTO subfeatures SET subfeature_type=?, subfeature_start=?, subfeature_end=?, ".
         "subfeature_chr=?, subfeature_strand=?, subfeature_phase=?, issue_id=?, created_on=?, updated_on=?;";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'subfeature_1'} = $sth;

  #for other things
  $sql = "UPDATE projects SET seen=? WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'seen_gene'} = $sth;

  $sql = "UPDATE issues SET seen=? WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'seen_transcript'} = $sth;

  $sql = "UPDATE subfeatures SET seen=? WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'seen_subfeature'} = $sth;

  $sql = "UPDATE flags SET seen=? WHERE id=?";
  my $dbh_seen_flags = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'seen_flags'} = $dbh_seen_flags;

  #check presence
  $sql   = "SELECT id FROM projects WHERE name=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'present_gene'} = $sth;

  $sql   = "SELECT id FROM issues WHERE subject=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'present_transcript'} = $sth;

  $sql   = "SELECT id, subfeature_start, subfeature_end FROM subfeatures ".
           "WHERE subfeature_chr=? AND issue_id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'present_subfeature'} = $sth;

  $sql   = "SELECT id FROM tecs ".
           "WHERE issue_id=? AND flag_id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'present_tec'} = $sth;

  $sql   = "SELECT id FROM tec_features ".
           "WHERE tec_id=? AND chromosome=? AND start=? AND end=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'present_tecfeature'} = $sth;

  #check coordinates
  $sql   = "SELECT Gstart, Gend, Gchrom, Gstrand  FROM projects WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'changed_gene'} = $sth;

  $sql   = "SELECT Tstart, Tend, Tchrom, Tstrand FROM issues WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'changed_transcript'} = $sth;

  $sql   = "SELECT subfeature_start, subfeature_end, subfeature_chr, subfeature_strand ".
           "FROM subfeatures WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'changed_subfeature'} = $sth;

  #check date
  $sql   = "SELECT updated_on FROM projects WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'changed_gene_date'} = $sth;

  $sql   = "SELECT updated_on FROM issues WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'changed_transcript_date'} = $sth;

  $sql   = "SELECT updated_on FROM subfeatures WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'changed_subfeature_date'} = $sth;

#  #check active state
#  $sql   = "SELECT is_public FROM projects WHERE id=?";
#  my $dbh_cactive_gene = $dbh->prepare($sql) or die $dbh->errstr;
#  $prepare_hash{'gene_active'} = $dbh_cactive_gene;
#  $sql   = "SELECT active FROM issues WHERE id=?";
#  my $dbh_cactive_transcript = $dbh->prepare($sql) or die $dbh->errstr;
#  $prepare_hash{'transcript_active'} = $dbh_cactive_transcript;
#  $sql   = "SELECT active FROM subfeatures WHERE id=?";
#  my $dbh_cactive_subfeature = $dbh->prepare($sql) or die $dbh->errstr;
#  $prepare_hash{'subfeature_active'} = $dbh_cactive_subfeature;

  $sql   = "UPDATE projects SET status_id=1 WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'gene_make_active'} = $sth;

  $sql   = "UPDATE issues SET status_id=1 WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'transcript_make_active'} = $sth;

  $sql   = "UPDATE subfeatures SET active=1 WHERE id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'subfeature_make_active'} = $sth;

  $sql = "UPDATE issues SET priority_id = ? WHERE id = ?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'set_priorities'} = $sth;

  $sql = "SELECT priority_id FROM issues WHERE id = ?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'get_priority'} = $sth;

  #$sql = "UPDATE issues SET active_flags = ? WHERE id = ?";
  #$sth = $dbh->prepare($sql) or die $dbh->errstr;
  #$prepare_hash{'active_flags'} = $sth;

  $sql = "SELECT cv.value FROM issues i, custom_fields cf, custom_values cv ".
         "WHERE i.id=? AND cv.customized_id=i.id AND cv.custom_field_id=cf.id AND cf.name=?";
  $sth = $dbh->prepare($sql);
  $prepare_hash{'custom_value'} = $sth;

  $sql = "UPDATE projects set updated_on=? where id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'set_gene_date'} = $sth;

  $sql = "UPDATE issues set updated_on=? where id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'set_transcript_date'} = $sth;

  $sql = "DELETE FROM subfeatures where issue_id=?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'subfeature_remove'} = $sth;

  #check parent
  $sql = "SELECT name FROM projects WHERE id = ?";
  $sth = $dbh->prepare($sql) or die $dbh->errstr;
  $prepare_hash{'project_name'} = $sth;

  $sql = "SELECT p.id, p.name, i.subject FROM projects p, issues i ".
            "WHERE p.id = i.project_id AND i.id = ?";
  $sth = $dbh->prepare($sql);
  $prepare_hash{'check_parent'} = $sth;

  $sql = "UPDATE issues SET project_id = ? WHERE id = ?";
  $sth = $dbh->prepare($sql);
  $prepare_hash{'update_parent'} = $sth;

  #get custom val
  $sql = 'select value from custom_values where customized_type=? and custom_field_id=? '.
         ' and customized_id=?';
  $sth = $dbh->prepare($sql);
  $prepare_hash{'check_custom'} = $sth;

  #insert custom val
  $sql = "INSERT INTO custom_values ".
         "SET customized_type=?, customized_id=?, custom_field_id=?, value=?";
  $sth = $dbh->prepare($sql);
  $prepare_hash{'insert_custom'} = $sth;

  #update custom val
  $sql = 'UPDATE custom_values SET value=? WHERE customized_type=? and custom_field_id=? '.
         ' and customized_id=?';
  $sth = $dbh->prepare($sql);
  $prepare_hash{'update_custom'} = $sth;

  #get field id of custom field
  $sql = "SELECT id FROM custom_fields WHERE name=?;";
  $sth = $dbh->prepare($sql);
  $prepare_hash{'get_custom_id'} = $sth;

  #delete custom val
  $sql = "DELETE FROM custom_values WHERE customized_type=? AND custom_field_id=? ".
         "AND customized_id=?";
  $sth = $dbh->prepare($sql);
  $prepare_hash{'delete_custom'} = $sth;

  #get secondary ids
  $sth = $dbh->prepare("SELECT target_id from id_conversions where source_id=?" );
  $prepare_hash{'id_conversion_1'} = $sth;
  $sth = $dbh->prepare("SELECT source_id from id_conversions where target_id=?" );
  $prepare_hash{'id_conversion_2'} = $sth;

  return(\%prepare_hash);
}


=head2 release_prepares

  Arg [1]    : hash with prepared statements
  Description: dispose the db handles
  Returntype : none

=cut

sub release_prepares {
    my ($prepare_hash) = @_;

    foreach my $pkey (keys %$prepare_hash){
      $prepare_hash->{$pkey}->finish;
    }
}


=head2 store_features

  Arg [1]    : db handle
  Arg [2]    : element to store
  Arg [3]    : type of element
  Arg [4]    : id to use 1
  Arg [5]    : id to use 2
  Arg [6]    : user id
  Arg [7]    : category id
  Description: entry function to store or update any feature in the GT db
  Returntype : int dbID,
               bool to indicate if feature was updated

=cut

sub store_features {
  my ($tracking_dbh, $prepare_hash, $element, $type, $useID_1, $useID_2, $user_id, $category_id) = @_;

  my $dbID  = 0;
  my $check = 0;
  my $updated = 1;

  if($VERBOSE){
    print "CHECKING with $type, $category_id, $user_id, $useID_1";
    if($useID_2){ print ", $useID_2" }
    print ".\n" ;
  }
  #does the identifier exists?
  $check = name_is_present($tracking_dbh, $prepare_hash, $type, $element, $useID_1, $useID_2);

  if( scalar @$check == 1){

    $dbID = $check->[0];

    #check activety status
#    if(!is_active($prepare_hash, $element, $dbID, $type)){
#      #re-activate!
#      make_active($tracking_dbh, $prepare_hash, $element, $dbID, $type, $user_id);
#    }
    if(get_status($tracking_dbh, $dbID, $type) eq "Rejected"){
      #re-activate!
      print "RE_ACTIVATING $type $dbID.\n";
      change_status($tracking_dbh, $dbID, 'Updated', $user_id, $type);
    }

    #use the last-mod date to skip old features
    my $datecheck = compare_dates($prepare_hash, $element, $dbID, $type);

    ### or force check #######################################################
    #my $datecheck = 1;

    if($datecheck){

      print ">DATE_CHANGED for $dbID\n" if($VERBOSE);

      #are the coordinates identical?
      my ($changes, $old_values) = compare_coordinates($tracking_dbh, $prepare_hash, $element, $dbID, $type);

      if(scalar @$changes){
	print ">UPDATING $dbID (".($element->{'id'})."): ".join(", ", @$changes)."\n";
	if($WRITE){ update_element($tracking_dbh, $element, $dbID, $changes, $user_id, $old_values, $type); }
      }
      else{
	$updated = 0;
	print ">no_loc_changes for $dbID\n" if($VERBOSE);
      }

      #compare parent id (for transcripts)?!
#      if($type eq "transcript"){
#	check_parent($tracking_dbh, $prepare_hash, $element, $dbID, $useID_1, $user_id);
#      }

      #compare type!
      if(exists($element->{'biotype'})){
	if(! update_custom_value($tracking_dbh, $prepare_hash, $element, $dbID, $user_id, "biotype") ){
	  print ">no_biotype_changes for $dbID\n" if($VERBOSE);
	}
      }

      #compare version!
      if(exists($element->{'version'})){
	if(! update_custom_value($tracking_dbh, $prepare_hash, $element, $dbID, $user_id, "version") ){
	  print ">no_version_changes for $dbID\n" if($VERBOSE);
	}
      }

      #compare status!
      if(exists($element->{'status'})){
	if(! update_custom_value($tracking_dbh, $prepare_hash, $element, $dbID, $user_id, "status") ){
	  print ">no_status_changes for $dbID\n" if($VERBOSE);
	}
      }

      #compare description!
      if(! update_element_description($tracking_dbh, $element, $dbID, $user_id) ){
	print ">no_description_changes for $dbID\n" if($VERBOSE);
      }

      #update date
      set_date($tracking_dbh, $prepare_hash, $dbID, $type, $element->{'updated_on'}, $user_id);

    }
    else{
      print ">no_date_changes for $dbID\n" if($VERBOSE);
      $updated = 0;
    }

    #set "seen" flag
    set_seen_flag($prepare_hash, $dbID, $type, "2");

  }
  elsif( scalar @$check > 1){
    #should not happen: non-unique id
    print "\nmultiple vals returned for query: ".join(", ", @$check)."!\n";
    return(0, 0);
  }
  else{
    #store new data
    print ">INSERTING $type, ".$element->{'chrom'}.", ".$element->{'start'}.", ".$element->{'end'} if $VERBOSE;
    print ", ".$element->{'id'} if(defined($element->{'id'}) and $VERBOSE);
    print "\n" if $VERBOSE;
    if($WRITE){ $dbID = insert_element($tracking_dbh, $prepare_hash, $element, $type, 
				       $useID_1, $useID_2, $user_id, $category_id); }
    $updated = 1;
    #set "seen" flag
    set_seen_flag($prepare_hash, $dbID, $type, "3");

  }

  return($dbID, $updated);
}


=head2 is_active

  Arg [1]    : db handle
  Arg [2]    : hash with db prepares
  Arg [3]    : element
  Arg [4]    : db ID
  Arg [5]    : type of element
  Description: check id existing element in db is marked as "active"
  Returntype : boolean, 1 if active

=cut

sub is_active {
  my ($prepare_hash, $element, $dbID, $type) = @_;

  warn "deprecated: feature not in use.\n";

  my $query_ref;
  if($type eq "gene"){
    $query_ref = $prepare_hash->{'gene_active'};
  }elsif($type eq "transcript"){
    $query_ref = $prepare_hash->{'transcript_active'};
  }elsif($type eq "subfeature"){
    $query_ref = $prepare_hash->{'subfeature_active'};
  }else{
    die "Can't check activety of type ". $type."!\n";
  }
  $query_ref->execute($dbID) or die "cant execute query!\n";

  #compare first part (date) of datestamps
  my ($active) = $query_ref->fetchrow_array;

  return $active;
}


=head2 make_active

  Arg [1]    : db handle
  Arg [2]    : hash with db prepares
  Arg [3]    : element
  Arg [4]    : db ID
  Arg [5]    : type of element
  Arg [6]    : user id
  Description: re-activate an existing element in db
  Returntype : none

=cut

sub make_active {
  my ($tracking_dbh, $prepare_hash, $element, $dbID, $type, $user_id) = @_;

  my $query_ref;
  if($type eq "gene"){
    $query_ref = $prepare_hash->{'gene_make_active'};
  }elsif($type eq "transcript"){
    $query_ref = $prepare_hash->{'transcript_make_active'};
  }elsif($type eq "subfeature"){
    $query_ref = $prepare_hash->{'subfeature_make_active'};
  }else{
    die "Can't make-active type ". $type."!\n";
  }
  $query_ref->execute($dbID) or die "cant execute query!\n";

  #write history
  my $notes = "RE-ACTIVATING $type $dbID!";
  write_history($tracking_dbh, $dbID, $type, $user_id, $notes);
}


=head2 compare_dates

  Arg [1]    : db handle
  Arg [2]    : hash with db prepares
  Arg [3]    : element
  Arg [4]    : db ID
  Arg [5]    : type of element
  Description: compare the last-mod datestamp
               of en element with the value
               currently found in the db
  Returntype : 1 if different, 0 if same

=cut

sub compare_dates {
  my ($prepare_hash, $element, $dbID, $type) = @_;

  #fetch last-mod date from db
  my $query_ref;
  if($type eq "gene"){
    $query_ref = $prepare_hash->{'changed_gene_date'};
  }elsif($type eq "transcript"){
    $query_ref = $prepare_hash->{'changed_transcript_date'};
  }elsif($type eq "subfeature"){
    $query_ref = $prepare_hash->{'changed_subfeature_date'};
  }else{
    die "Can't check ID of type ". $type."!\n";
  }
  $query_ref->execute($dbID);

  #compare first part (date) of datestamps
  my ($old_date) = $query_ref->fetchrow_array;
  my $new_date = $element->{'updated_on'};
  if(!($old_date && $new_date)){
    return 1;
  }
  $old_date =~ /^(\d{4})\-(\d{2})\-(\d{2}).*$/;
  $old_date = $1.'/'.$2.'/'.$3;
  $new_date =~ /^(\d{4})\-(\d{2})\-(\d{2}).*$/;
  $new_date = $1.'/'.$2.'/'.$3;

  my $cmpflag = $old_date cmp $new_date;

  if($cmpflag > 0){
    #date in TrackT is NEWER than the one in the DAS response
    print "DATE MIXUP $old_date / $new_date ($type: $dbID)!\n";
    return 1;
  }
  elsif($cmpflag < 0){
    #needs updating
    return 1;
  }

  return 0;
}


=head2 get_project_for_category

  Arg [1]    : db handle
  Arg [2]    : category id
  Arg [3]    : category name
  Description: get the id of the project using either
               the category id or name.
               Only for categories where there is one
               "general" project for all issues.
  Returntype : int project_id

=cut


sub get_project_for_category {
  my ($prepare_hash, $dbh, $category_id, $category_name) = @_;

  my $project_id = 0;
  my $c = 0;
  my ($sql, $queryterm);
  if($category_id){
    $sql = "SELECT DISTINCT p.id FROM projects p, issues i WHERE p.id=i.project_id AND i.category_id=?";
    $queryterm = $category_id;
  }
  elsif($category_name){
    $sql = "SELECT DISTINCT p.id FROM projects p, issues i, issue_categories ic WHERE p.id=i.project_id ".
           "AND i.category_id=ic.id AND ic.name=?";
    $queryterm = $category_name;
  }
  else{
    print STDERR "Missing category id or name.\n";
    return 0;
  }
  my $sth = $dbh->prepare($sql);
  $sth->execute($queryterm);
  while(my ($id) = $sth->fetchrow_array()){
    $project_id = $id;
    if($c++>1){
      print STDERR "Multiple projects found.\n";
      return 0;
    }
  }

  set_seen_flag($prepare_hash, $project_id, "gene", "2");

  return $project_id;
}


=head2 create_general_project

  Arg [1]    : db handle
  Arg [2]    : category name
  Description: create a "general" project for all issues of a category.
               This is much more efficient for entries that don't need
               a gene-transcript relationship anyway
  Returntype : string project_name

=cut

sub create_general_project {
  my ($dbh, $category_name) = @_;

  my $project_id = 0;
  if($category_name){
    my $sql = "INSERT INTO projects SET name=?, description=?, created_on=NOW(), updated_on=NOW(), 
              identifier=?, Gchrom='0', Gstart=0, Gend=0, Gstrand='+'";
    my $sth = $dbh->prepare($sql);
    if($WRITE){
      $sth->execute($category_name, "General parent-entry for entries $category_name", $category_name)
	or die "can't exeute insert.\n";
      $project_id = $dbh->last_insert_id(undef, undef, undef, undef);
    }
  }
  else{
    print STDERR "Missing category name.\n";
    return 0;
  }

  return $project_id;
}


=head2 check_parent

  Arg [1]    : db handle
  Arg [2]    : hash with db prepares
  Arg [3]    : element
  Arg [4]    : element db ID
  Arg [5]    : parent db ID
  Arg [6]    : user id
  Description: compare the parent (gene) name
               of en element (transcript) with the value
               currently found in the db, update if necessary
  Returntype : 1 if different, 0 if same

=cut

sub check_parent {
  my ($dbh, $prepare_hash, $element, $this_id, $geneID, $user_id) = @_;

  my ($note);
  my $update = 0;
  print "Checking parent of $this_id, gene=$geneID\n" if($VERBOSE);
  #compare names of the transcript's parent gene
  if(!exists($element->{'parent'})){
    print "No parent defined for ".$element->{'id'}."!!!\n";
    return 0;

    #TODO: check if partent is defined in db...
    #$note = "Removed parent XYZ";
    #$update = 1;
  }

  #get real id of parent
  my $parent_name = _check_id($element->{'parent'});
  #get db entry of parent
  $prepare_hash->{'check_parent'}->execute($this_id);
  my ($db_gene_id, $gene_name, $t_name) = $prepare_hash->{'check_parent'}->fetchrow_array();

  if(!defined($gene_name)){
    print "Parent of ".$element->{'id'}." ($this_id) not defined in database!\n";
    #get db id of parent entry
    $prepare_hash->{'present_gene'}->execute($parent_name);
    my $db_parent_id = $prepare_hash->{'present_gene'}->fetchrow_array();

    if(!defined($db_parent_id)){
      print "Could not find parent!!!\n";
      return 0;
    }
    #update!
    $prepare_hash->{'update_parent'}->execute($db_parent_id, $this_id);

    $note = "Added parent ".$parent_name;
    $update = 1;
  }
  elsif($gene_name ne $parent_name){
    print "Parent ($t_name) defined differently in database 1 (db:".$gene_name." / source:".$parent_name.")\n";
    #update!
    #get db id of parent entry
    $prepare_hash->{'present_gene'}->execute($element->{'parent'});
    my $db_parent_id = $prepare_hash->{'present_gene'}->fetchrow_array();
    $prepare_hash->{'update_parent'}->execute($db_parent_id, $this_id);

    $note = "Changed parent from $gene_name to ".$parent_name;
    $update = 1;
  }
  elsif($db_gene_id != $geneID){
    print "Parent ($t_name) defined differently in database 2 (db:".$db_gene_id." / source:".$geneID.")\n";
    #does parent exists?

    #update!
    #$prepare_hash{'update_parent'}->execute($element->{'parent'}, $this_id);

    $note = "Changed parent from $gene_name to ".$parent_name;
    $update = 1;
  }

  if($update){
    #write history
    write_history($dbh, $this_id, $element->{'type'}, $user_id, $note);
  }

  return $update;
}


=head2 insert_element

  Arg [1]    : element to store
  Arg [2]    : type of element
  Arg [3]    : id to use 1
  Arg [4]    : id to use 2
  Arg [5]    : ref to hash with prepared statements
  Arg [6]    : db handle
  Arg [7]    : user id
  Arg [8]    : category id
  Description: store new element in db
  Returntype : new db id

=cut

sub insert_element {
  my ($tracking_dbh, $prepare_hash, $element, $type, $useID_1, $useID_2, $user_id, $category_id) = @_;

  my $dbID = 0;

  if($type eq "gene") {
    #tables for genes: project, custom_values, project_trackers
    $dbID = _store_gene($tracking_dbh, $prepare_hash, $element, $user_id, $category_id);
    print "INSERTED gene $dbID\n\n" if($VERBOSE);
  }
  elsif($type eq "transcript") {
    #tables for transcripts: issues, issue_categories, scores, custom_values
    $dbID = _store_transcript($tracking_dbh, $prepare_hash, $element, $user_id, $useID_1, $category_id);
    print "INSERTED transcript $dbID\n\n" if($VERBOSE);
  }
  else {
    #tables for subelements: subfeatures
    $dbID = _store_feature($tracking_dbh, $prepare_hash, $element, $user_id, $useID_1, $useID_2);
    print "INSERTED subfeature $dbID\n\n" if($VERBOSE);
  }

  return $dbID;
}


=head2 _get_custom_value_ids

  Arg [1]    : db handle
  Description: get db ids for all custom values, eg. 'transcript_type'
  Returntype : ref to hash with db ids for all custom values defined

=cut

sub _get_custom_value_ids {
  my ($dbh) = @_;

  my $role_id    = 5;
  my $my_id      = 1;
  my $my_role_id = 3;
  my %custom_fields = (
		       'gene_type'           => 4,
		       'gene_status'         => 5,
		       'gene_version'        => 12,
		       'transcript_type'     => 8,
		       'transcript_status'   => 9,
		       'transcript_version'  => 11,
		      );

  my $sql = "SELECT id FROM custom_fields WHERE name=?";
  my $sth = $dbh->prepare($sql);
  foreach my $name (keys %custom_fields){
    $sth->execute($name);
    my ($dbID) = $sth->fetchrow_array;
    if(!$dbID){
      warn "No id defined for custom field $name!\n";
      $dbID = 0;
    }
    $custom_fields{$name} = $dbID;
  }

  return \%custom_fields;
}


=head2 _store_gene

  Arg [1]    : db handle
  Arg [2]    : ref to hash with prepared statements
  Arg [3]    : gene to store
  Arg [4]    : use id
  Description: write a new gene (as project) to the db
               private method
  Returntype : new db id

=cut

sub _store_gene {
  my ($dbh, $prepare_hash, $gene, $user_id, $category_id) = @_;

  my $tracker_id = get_tracker_id($TRACKER, $dbh);
  my @enabled_modules = qw( issue_tracking wiki score_module );
  my $dbID = 0;
  #these values should be fetched dynamically at one point:
  my $role_id    = 5;
  my $my_id      = 1;
  my $my_role_id = 3;
  my %custom_values = (
		       'gene_type'           => 4,
		       'gene_status'         => 5,
		       'gene_version'        => 12,
		       'transcript_type'     => 8,
		       'transcript_status'   => 9,
		       'transcript_version'  => 11,
		      );

  #check strand
  $gene->{'strand'} = _check_strand($gene->{'strand'});

  #check/clean id
  my $clean_id = _check_id($gene->{'id'});

  #check creation data
  my $creation = _check_date($gene->{'created_on'}, 1);
  #use creation for update date?
  if(!($gene->{'updated_on'})){
    $gene->{'updated_on'} = $creation;
  }
  else{
    #check format
  }

  #store gene
  $prepare_hash->{'gene_1'}->execute( $clean_id, $gene->{'description'}, $creation, $gene->{'updated_on'},
				      lc($clean_id), $gene->{'chrom'}, $gene->{'start'}, $gene->{'end'},
				      $gene->{'strand'} ) 
    or die "cant insert gene 1!\n";
  $dbID = $dbh->last_insert_id(undef, undef, undef, undef);

  #store tracker-relation
  $prepare_hash->{'gene_2'}->execute( $dbID, $tracker_id )or die "cant insert gene 2!\n";
  if($gene->{'status'}){
    $prepare_hash->{'gene_3'}->execute( 'Project', $dbID, $custom_values{'gene_status'}, $gene->{'status'} )
      or die "cant insert gene 3!\n";
  }
  if($gene->{'biotype'}){
    $prepare_hash->{'gene_3'}->execute( 'Project', $dbID, $custom_values{'gene_type'}, $gene->{'biotype'} )
      or die "cant insert gene 3!\n";
  }
  #author info
  $prepare_hash->{'gene_4'}->execute( $user_id, $role_id, $dbID ) or die "cant insert gene 4!\n";
  #add me as Master
  $prepare_hash->{'gene_4'}->execute( $my_id, $my_role_id, $dbID ) or die "cant insert gene 4!\n";

  #enable modules
  foreach my $module_name (@enabled_modules) {
     $prepare_hash->{'gene_5'}->execute( $dbID, $module_name ) or die "cant insert gene 5!\n";
  }

  #add category
  #$prepare_hash->{'gene_6'}->execute( $dbID, $category_id ) or die "cant insert gene 6!\n";

  return $dbID;
}


=head2 _store_transcript

  Arg [1]    : gene to store
  Arg [2]    : ref to hash with prepared statements
  Arg [3]    : db handle
  Arg [4]    : use id
  Description: write a transcript (as an issue) to the db
               private method
  Returntype : new db id

=cut

#write a transcript (as an issue) to the db
sub _store_transcript {
  my ($dbh, $prepare_hash, $transcript, $user_id, $gene_id, $category) = @_;

  my $tracker_id = get_tracker_id($TRACKER, $dbh);
  my $status     = $STATUS{'New'};
  my $priority   = $PRIORITY{'normal'};
  my $dbID = 0;

  #these values should be fetched dynamically at one point:
  my %custom_values = (
      'transcript_type'       => 8,
      'transcript_status'     => 9,
      'transcript_version'    => 11,
      );

  #check strand
  $transcript->{'strand'} = _check_strand($transcript->{'strand'});

  #check/clean id
  my $clean_id = _check_id($transcript->{'id'});

  #check creation date
  my $creation = _check_date($transcript->{'created_on'}, 1);
  #use creation for update date?
  if(!($transcript->{'updated_on'})){
    $transcript->{'updated_on'} = $creation;
  }
  else{
    #check format
  }

  #store transcript
  $prepare_hash->{'transcript_1'}->execute( $tracker_id, $gene_id, $clean_id, $transcript->{'description'},
					    $category, $status, $priority, $user_id, $creation,
					    $transcript->{'updated_on'}, $transcript->{'chrom'}, $transcript->{'start'},
					    $transcript->{'end'}, $transcript->{'strand'} )
    or die "cant insert transcript!\n";
  $dbID = $dbh->last_insert_id(undef, undef, undef, undef);

  if($transcript->{'biotype'}){
    $prepare_hash->{'transcript_3'}->execute( 'Issue', $dbID, $custom_values{'transcript_type'},   $transcript->{'biotype'} ) or die "cant insert transcript biotype!\n";
  }
  if($transcript->{'status'}){
    $prepare_hash->{'transcript_3'}->execute( 'Issue', $dbID, $custom_values{'transcript_status'}, $transcript->{'status'} ) or die "cant insert transcript status!\n";
  }
  if($transcript->{'version'}){
    $prepare_hash->{'transcript_3'}->execute( 'Issue', $dbID, $custom_values{'transcript_version'}, $transcript->{'version'} ) or die "cant insert transcript version!\n";
  }

  return $dbID;
}



=head2 _store_feature

  Arg [1]    : feature to store
  Arg [2]    : ref to hash with prepared statements
  Arg [3]    : db handle
  Arg [4]    : use id
  Description: write a feature (exon, etc.) to the db
               private method
  Returntype : new db id

=cut

sub _store_feature {
  my ($dbh, $prepare_hash, $feature, $user_id, $gene_id, $transcript_id) = @_;

  my $dbID = 0;

  $feature->{'strand'} = _check_strand($feature->{'strand'});

  #check creation data
  my $creation = _check_date($feature->{'created_on'}, 1);
  #use creation for update date?
  if(!($feature->{'updated_on'})){
    $feature->{'updated_on'} = $creation;
  }
  else{
    #check format
  }
  my $phase = ".";
  if(exists($feature->{'phase'})){
    $phase = $feature->{'phase'};
  }
#  if($phase eq "-"){
#    $phase = ".";
#  }
  #store features
  $prepare_hash->{'subfeature_1'}->execute( $feature->{'type'}, $feature->{'start'}, $feature->{'end'},
					    $feature->{'chrom'}, $feature->{'strand'}, $phase,
					    $transcript_id, $creation, $feature->{'updated_on'} )
 	or die "cant insert feature 1!\n";
  $dbID = $dbh->last_insert_id(undef, undef, undef, undef);

  return $dbID;
}


=head2 remove_subfeatures

  Arg [1]    : db handle
  Arg [2]    : ref to hash with prepared statements
  Arg [3]    : transcript / issue db id
  Description: delete all subfeatures belonging to a certain ranscript / issue
  Returntype : none

=cut

sub remove_subfeatures {
  my ($tracking_dbh, $prepare_hash, $current_transcript_id) = @_;

  if($WRITE){
    $prepare_hash->{'subfeature_remove'}->execute( $current_transcript_id );
  }
}


=head2 print_element

  Arg [1]    : ref to hash
  Description: print out feature attributes for testing
  Returntype : none

=cut

sub print_element {
  my ($element) = @_;

  foreach my $key (%{$element}){
    if($key and exists($element->{$key})){
      print "\t".$key.": ";
      if(defined($element->{$key})){
	print $element->{$key}."\n";
      }
      else{
	print "--\n";
      }
    }
  }
  print "\n";
#  print $element->{'chrom'}."\t";
#  print $element->{'source'}."\t";
#  print $element->{'type'}."\t";
#  print $element->{'start'}."\t";
#  print $element->{'end'}."\t";
#  print $element->{'score'}."\t";
#  print $element->{'strand'}."\t";
#  print $element->{'phase'}."\t";
#  print $element->{'attributes'}."\n";

}


=head2 add_category

  Arg [1]    : db handle
  Arg [2]    : category name
  Arg [3]    : source description
  Description: store a new issue category
  Returntype : new db id

=cut

sub add_category {
  my ($tracking_dbh, $category_name, $description, $create_default_project) = @_;

  my $sql = "INSERT INTO issue_categories SET project_id=-1, name=?, description=?;";
  my $sth = $tracking_dbh->prepare($sql);
  $sth->execute($category_name, $description) or die "cant insert  category $category_name!\n";
  my $dbID = $tracking_dbh->last_insert_id(undef, undef, undef, undef);
  $sth->finish;

  if($create_default_project){
    #create "general" project level entry
    create_general_project($tracking_dbh, $category_name);
  }

  return $dbID;
}


=head2 name_is_present

  Arg [1]    : db handle
  Arg [2]    : hash with db handles
  Arg [3]    : type of element
  Arg [4]    : element to check
  Arg [5]    : id 1
  Arg [6]    : id 2
  Description: check by name/subject whether an object is stored in the db
  Returntype : db id if found

=cut

sub name_is_present {
  my ($tracking_dbh, $prepare_hash, $type, $element, $id1, $id2) = @_;

  my $dbID      = 0;
  my $clean_id  = "";
  my @returnIDs = ();
  my $c         = 0;
  my $projectID = 0;
  my ($table, $sth);

##############################################################
#return \@returnIDs;
##############################################################


  if($type eq "gene"){
    #check/clean id
    $clean_id = _check_id($element->{'id'});
    $prepare_hash->{'present_gene'}->execute( $clean_id )
      or die "Cant execute query to check gene.\n";
    while(($dbID) = $prepare_hash->{'present_gene'}->fetchrow_array){
      push(@returnIDs, $dbID);
      if($c++ > 1){ warn "ID $dbID ($type) appears more than once!"; };
    }
  }elsif($type eq "transcript"){
    #ADD project id or other id?
    #check/clean id
    $clean_id = _check_id($element->{'id'});
    $prepare_hash->{'present_transcript'}->execute( $clean_id )
      or die "Cant execute query to check transcript.\n";
    while(($dbID) = $prepare_hash->{'present_transcript'}->fetchrow_array){
      push(@returnIDs, $dbID);
      if($c++ > 1){ warn "ID $dbID ($type) appears more than once!\n"; };
    }

    #warn "OK: ".$element->{'id'}."\n";

  }elsif($type eq "subfeature"){

##############################################################
    return \@returnIDs;
##############################################################

    #for subfeatures:
    #get all belonging to this transcript
    # -compare borders
    $prepare_hash->{'present_subfeature'}->execute( $element->{'chrom'}, $id2 )
      or die "Cant execute query to check subfeatures.\n";

    #print "E: ".$element->{'start'}.", ".$element->{'end'}.", ".
    #      $element->{'chrom'}.", ".$element->{'strand'}.", $id2\n";

    #find the ones overlapping this feature
    my ($sstart, $send);
    while(($dbID, $sstart, $send) = $prepare_hash->{'present_subfeature'}->fetchrow_array){
      if( ($element->{'start'} <= $send) and ($element->{'end'} >= $sstart) ){
	push(@returnIDs, $dbID);
      }
    }
  }elsif($type eq "tec_feature"){
    #check/clean id
    my $tec_id = _check_id($element->{'tec_id'});
    $prepare_hash->{'present_tecfeature'}->execute( $tec_id, $element->{'chrom'},
						    $element->{'start'}, $element->{'end'} )
      or die "Cant execute query to check tec-feature.\n";
    while(($dbID) = $prepare_hash->{'present_tecfeature'}->fetchrow_array){
      push(@returnIDs, $dbID);
      print "FOUND TF: $tec_id, ".$element->{'start'}.", $dbID\n" if $VERBOSE;
      if($c++ > 1){ warn "ID $dbID ($type) appears more than once!"; };
    }
  }elsif($type eq "tec"){
    #check/clean id
    my $issue_id = _check_id($element->{'issue_id'});
    my $flag_id  = _check_id($element->{'flag_id'});
    $prepare_hash->{'present_tec'}->execute( $issue_id, $flag_id )
      or die "Cant execute query to check tec.\n";
    while(($dbID) = $prepare_hash->{'present_tec'}->fetchrow_array){
      push(@returnIDs, $dbID);
      if($c++ > 1){ warn "ID $dbID ($type) appears more than once!"; };
    }
  }else{
    die "Can't check ID of type $type.\n";
  }

#  if($VERBOSE){
#    print "checked $type, ".$element->{'chrom'}.", ".$element->{'start'}.", ".$element->{'end'};
#    print ", ".$element->{'id'} if(defined($element->{'id'}));
#    print " (1=$id1)" if($id1);
#    print " (2=$id2)" if($id2);
#    print " => id => ".join(", ", @returnIDs) if(scalar@returnIDs );
#    print "\n";
#  }

  return \@returnIDs;
}


=head2 compare_coordinates

  Arg [1]    : db handle
  Arg [2]    : hash with db handles
  Arg [3]    : element
  Arg [4]    : db ID
  Description: compare coordinates (start, end, strand, type)
               of en element with the values
               currently found in the db
  Returntype : array ref with types of changed fields

=cut

#are the coordinates identical?
sub compare_coordinates {
  my ($tracking_dbh, $prepare_hash, $element, $dbID, $type) = @_;

  my $query_ref;

  if($type eq "gene"){
    $query_ref = $prepare_hash->{'changed_gene'};
  }elsif($type eq "transcript"){
    $query_ref = $prepare_hash->{'changed_transcript'};
  }elsif($type eq "subfeature"){
    $query_ref = $prepare_hash->{'changed_subfeature'};
  }else{
    die "Can't check ID of type ". $type."!\n";
  }

  $query_ref->execute($dbID);

  my ($changes, $old_values) = _do_compare($query_ref, $element);

  return($changes, $old_values);
}


=head2 _do_compare

  Arg [1]    : db query handle
  Arg [2]    : element
  Description: do the acutal value comparisome for
               the function compare coordinates
  Returntype : none

=cut

sub _do_compare {
  my ($query_ref, $element) = @_;

  my @changes = ();
  my @old_values = ();

  while(my ($start, $end, $chrom, $strand, $type) = $query_ref->fetchrow_array){

    if($element->{'start'} != $start){
      push(@changes, 'start');
      push(@old_values, $start);
    }
    if($element->{'end'} != $end){
      push(@changes, 'end');
      push(@old_values, $end);
    }
    if((_check_strand($element->{'strand'})) ne _check_strand($strand)){

      print "STRANDCHECK ".($element->{'strand'})." ne $strand.\n";

      push(@changes, 'strand');
      push(@old_values, $strand);
    }
    if($element->{'chrom'} ne $chrom){
      print "FOUND DIFFS: ".$element->{'chrom'}." != $chrom.\n";
      push(@changes, 'chrom');
      push(@old_values, $chrom);
    }
#    if($element->{'type'} != $type){
#      push(@changes, 'type');
#      push(@old_values, 'type');
#    }
  }

  return(\@changes, \@old_values);
}


=head2 update_element

  Arg [1]    : db query handle
  Arg [2]    : element
  Description: update an element
  Returntype : none

=cut

sub update_element {
  my ($dbh, $element, $dbID, $changes, $user_id, $old_values, $type) = @_;

  my ($sql, $notes, $table);
  my $c = 0;

  #check type
  # (requires some more abstraction or def in config)
  my %fields = (
		'gene' => {
			   'start'  => 'Gstart',
			   'end'    => 'Gend',
			   'strand' => 'Gstrand',
			   'chrom'  => 'Gchrom',
			   'type'   => '',
			   },
		'transcript' => {
			   'start'  => 'Tstart',
			   'end'    => 'Tend',
			   'strand' => 'Tstrand',
			   'chrom'  => 'Tchrom',
			   'type'   => '',
			   },
		'subfeature' => {
			   'start'  => 'subfeature_start',
			   'end'    => 'subfeature_end',
			   'strand' => 'subfeature_strand',
			   'phase'  => 'subfeature_phase',
			   'chrom'  => 'subfeature_chr',
			   'type'   => 'subfeature_type',
			   },
	       );

  #table name mappings
  if($type eq "gene"){
    $table = "projects";
  }elsif($type eq "transcript"){
    $table = "issues";
  }elsif($type eq "subfeature"){
    $table = "subfeatures";
  }else{
    die "Can't update type $type.\n";
  }

  foreach my $change (@$changes){
    #prepare dynamic sql statement
    my $sql = "UPDATE ".$table." SET ".$fields{$type}{$change}." = ? WHERE id = ?";
    my $sth = $dbh->prepare($sql);
    #update field
    $sth->execute($element->{$change}, $dbID);
    $sth->finish;
    #write history
    $notes = "Updated ".$change." from ".(${$old_values}[$c])." to ".$element->{$change};
    write_history($dbh, $dbID, $type, $user_id, $notes);
    $c++;
  }

}


=head2 update_custom_value

  Arg [1]    : db handle
  Arg [2]    : db prepare_hash
  Arg [3]    : element
  Arg [4]    : dbID of the affected element
  Arg [5]]   : user ID
  Arg [5]]   : name of the field to check
  Description: compare/update the custom value of an element
  Returntype : boolean, true if db value was changed

=cut

sub update_custom_value {
  my ($tracking_dbh, $prepare_hash, $element, $dbID, $user_id, $checkfield) = @_;

  die "\n---------------\nMissing ID from ".$element->{'id'}."\n---------------\n" if(!$dbID);

  my $type = $element->{'type'};
  my $rtype;
  if($type eq "gene"){
    $rtype = "Project";
  }
  elsif($type eq "transcript"){
    $rtype = "Issue";
  }else{
    print "notchecking ".$type."\n";
    return 0;
  }
  my $changed = 0;

  #define relation between source-field names and database custom-field names
  # (requires some more abstraction or def in config)
  my %fields = (
		'biotype'           => $type.'type',
		'version'           => $type.'version',
		'status'            => $type.'status',
	       );

  my $fieldname = $fields{$checkfield} or die "Cant check custom field $checkfield.\n";
  #print STDERR " CHECKING ".$rtype." and ".$checkfield." of ".$element->{'id'}."\n" if($VERBOSE);

  #get the custom field ids for the field to check
  $prepare_hash->{'get_custom_id'}->execute($fieldname);
  my ($field_id) = $prepare_hash->{'get_custom_id'}->fetchrow_array();
  if(!$field_id){
    die "Cant check custom field $fieldname.\n";
  }

  #get the custom value
  $prepare_hash->{'check_custom'}->execute($rtype, $field_id, $dbID);
  my ($field_value) = $prepare_hash->{'check_custom'}->fetchrow_array();
  if(!$field_value){
    $field_value = 0;
  }

  #print "CHECKING ".($element->{$checkfield})." and ".$field_value."\n" if($VERBOSE);

  #compare new & old value
  if(!exists($element->{$checkfield}) and defined($element->{$checkfield}) and $field_value){
    print ">Removing custom_value $field_value [$type $dbID]\n" if($VERBOSE);
    if($WRITE){
      $prepare_hash->{'delete_custom'}->execute($rtype, $field_id, $dbID);
      my $notes = "Changed $checkfield from $field_value to ".$element->{$checkfield};
      write_history($tracking_dbh, $dbID, $type, $user_id, $notes);
    }
    $changed = 1;
  }
  elsif(exists($element->{$checkfield}) and defined($element->{$checkfield}) and !$field_value){
    print ">Inserting custom_value ".$element->{$checkfield}." [$type $dbID]\n" if($VERBOSE);
    if($WRITE){
      $prepare_hash->{'insert_custom'}->execute($rtype, $dbID, $field_id,
						$element->{$checkfield});
    }
    $changed = 1;
  }
  elsif(exists($element->{$checkfield}) and defined($element->{$checkfield}) and $field_value){
    print "Comparing custom_value ".$element->{$checkfield}." and ".
                 $field_value." [$type $dbID]\n" if($VERBOSE);
    if(($element->{$checkfield}) ne $field_value){
      print ">Update $fieldname from $field_value to ".$element->{$checkfield}.".\n" if($VERBOSE);
      if($WRITE){
	#update & write history
	$prepare_hash->{'update_custom'}->execute($element->{$checkfield},
						  $rtype, $field_id, $dbID);
	my $notes = "Changed $checkfield from $field_value to ".$element->{$checkfield};
	write_history($tracking_dbh, $dbID, $type, $user_id, $notes);
      }
      $changed = 1;
    }
  }
  else{
    print STDERR "1:".$rtype."\n";
    print STDERR "2:".$checkfield."\n";
    print STDERR "3:".$element->{'id'}."\n";
    print STDERR "4:".$element->{$checkfield}.".\n";
    print STDERR "5:".$field_value.".\n";
    warn "QUE?!\n";
  }

  return $changed;
}


=head2 update_element_description

  Arg [1]    : db query handle
  Arg [2]    : element
  Arg [3]    : dbID of the affected element
  Arg [4]]   : user ID
  Description: compare/update the description of an element
  Returntype : none

=cut


sub update_element_description {
  my ($tracking_dbh, $element, $dbID, $user_id) = @_;

  my $changed = 0;
  my ($sql, $sth, $table);
  my $type = $element->{'type'};

  if($type eq "gene"){
    $table = "projects";
  }
  elsif($type eq "transcript"){
    $table = "issues";
  }
  else{
    print "Cant check description of $type.\n";
    return 0;
  }
  #check description
  if(!($element->{'description'})){
    print "No description for $type $dbID!\n";
    return 0;
  }
  $sql = 'SELECT description FROM '.$table.' WHERE id=?';
  $sth = $tracking_dbh->prepare($sql);
  $sth->execute($dbID);
  my ($description) = $sth->fetchrow_array();

  #compare old & new type
  if((!$description && ($element->{'description'})) or ($description ne $element->{'description'})){
    #insert & write history?
    if(!$description){ $description = '-' }
    print ">>updating description from $description to ".($element->{'description'})."\n" if($VERBOSE);
    if($WRITE){
      $sql = 'UPDATE '.$table.' SET description=? where id=?';
      $sth = $tracking_dbh->prepare($sql);
      $sth->execute($element->{'description'}, $dbID) or warn "Cant update description for $dbID.\n";

      my $notes = "Changed description: $description";
      write_history($tracking_dbh, $dbID, $type, $user_id, $notes);
    }
    $changed = 1;
  }
  $sth->finish;

  return $changed;
}


=head2 set_seen_flag

  Arg [1]    : hash with db handles
  Arg [2]    : element
  Arg [3]    : type of element
  Arg [4]    : value for flag
  Description: set a flag for housekeeping
  Returntype : none

=cut

sub set_seen_flag {
  my ($prepare_hash, $element_id, $type, $value) = @_;

  if(($type eq "gene") or ($type eq "project")){
    $prepare_hash->{'seen_gene'}->execute( $value, $element_id );
  }
  elsif(($type eq "transcript") or ($type eq "issue")){
    $prepare_hash->{'seen_transcript'}->execute( $value, $element_id );
  }
  elsif($type eq "subfeature"){
    $prepare_hash->{'seen_subfeature'}->execute( $value, $element_id );
  }
  elsif($type eq "flag"){
    $prepare_hash->{'seen_flags'}->execute( $value, $element_id );
  }
  else{
    die "Can't set flag of type $type.\n";
  }

}


=head2 reset_seen_flag

  Arg [1]    : hash with db handles
  Arg [2]    : type to re-set (projects, issues or subfeatures)
  Arg [3]    : category to reset (eg. Havana)
  Description: set flags for housekeeping back to "0"
  Returntype : none

=cut

sub reset_seen_flags {
  my ($dbh, $type, $category) = @_;

  my ($sql, $sth);

  if($category && !$type){
#    $sql = "UPDATE projects p, issues i, subfeatures sf, issue_categories ic ".
#           "SET p.seen = 0, i.seen = 0, sf.seen = 0 ".
#           "WHERE p.id = i.project_id AND i.id = sf.issue_id AND i.category_id = ic.id AND ic.name = ? ";
    #projects
    $sql = "UPDATE projects p left join issues i on p.id = i.project_id ".
           "LEFT JOIN issue_categories ic ON i.category_id = ic.id ".
           "SET p.seen = 0 WHERE ic.name = ? ";
    $sth = $dbh->prepare($sql);
    $sth->execute($category);
    #issues
    $sql = "UPDATE issues i LEFT JOIN issue_categories ic ON i.category_id = ic.id ".
           "SET i.seen = 0 WHERE ic.name = ? ";
    $sth = $dbh->prepare($sql);
    $sth->execute($category);
    #subfeatures?

  }
  elsif($type && $type eq "flags"){
    $sql = "UPDATE flags SET seen = 0 ";
    if($category){
      $sql .= " AND flag_name = ?";
      $sth = $dbh->prepare($sql);
      $sth->execute($category);
    }
    else{
      $sth = $dbh->prepare($sql);
      $sth->execute();
    }
  }
  elsif($type && $type eq "issues"){
    $sql = "UPDATE issues SET seen = 0 ";
    if($category){
      $sql .= " AND flag_name = ?";
      $sth = $dbh->prepare($sql);
      $sth->execute($category);
    }
    else{
      $sth = $dbh->prepare($sql);
      $sth->execute();
    }
  }
  else{
    print STDERR "RE-SET ALL SEEN FLAGS???\n";
#    foreach my $table qw(projects issues subfeatures){
#      $sql = "UPDATE ".$table." SET seen = 0";
#      $dbh->do($sql);
#    }
  }

}


=head2 analyse_seen_flags

  Arg [1]    : hash with db handles
  Arg [2]    : type to re-set (projects, issues, subfeatures)
  Arg [3]    : category to reset (Havana)
  Description: check with entries where not visited during update
               and retire these entries:
               set status to rejected, write history entry
  Returntype : none

=cut

sub analyse_seen_flags {
  my ($dbh, $type, $category, $user_id) = @_;

  if(!$WRITE){ return 0; }

  my ($sql, $sth);

  if($category && !$type){
    my $retired_p = 0;
    my $retired_i = 0;
    print "Analysing flags....\n" if($VERBOSE);
    #projects
    $sql = "SELECT DISTINCT p.id, p.name FROM projects p, issues i, issue_categories ic, issue_statuses iss ".
           "WHERE p.id = i.project_id AND p.seen = 0 AND i.category_id = ic.id AND ic.name = ? ".
	   "AND p.status_id=iss.id AND iss.name != \"".$REJECTED_STATUS."\"";
    $sth = $dbh->prepare($sql);
    $sth->execute($category);
    while(my ($p_id, $p_name) = $sth->fetchrow_array){
      print "RETIRING PROJECT $p_id, $p_name.\n";
      if($WRITE){
	change_status($dbh, $p_id, $REJECTED_STATUS, $user_id, 'gene');
#      if($category eq "HAVANA"){
#	#set off alarm!
#	my $text    = "PROJECT $p_id OF CATEGORY $category WAS NOT FOUND ANYMORE!\n";
#	my $uemail  = $ALERT_EMAIL;
#	my $myemail = 'AnnoTrack@sanger.ac.uk';
#	my $subj    = "RETIRING AnnoTrack PROJECT";
#	#send_mail($text, $uemail, $myemail, $subj);
#      }
      }
      $retired_p++;
    }

    #issues
    $sql = "SELECT i.id, i.subject FROM issues i, issue_categories ic, issue_statuses iss ".
           "WHERE i.category_id = ic.id AND i.seen = 0 AND ic.name = ? ".
	   "AND i.status_id=iss.id AND iss.name != \"".$REJECTED_STATUS."\"";
    $sth = $dbh->prepare($sql);
    $sth->execute($category);
    while(my ($i_id, $i_name) = $sth->fetchrow_array){
      print "RETIRING ISSUE $i_id, $i_name.\n";
      if($WRITE){
	change_status($dbh, $i_id, $REJECTED_STATUS, $user_id, 'transcript');
      }
      $retired_i++;
    }

#    #subfeats
#    $sql = "SELECT sf.id FROM subfeatures sf, issues i, issue_categories ic ".
#           "WHERE sf.issue_id = i.id AND i.category_id = ic.id AND sf.seen = 0 AND ic.name = ? ".
#	   "AND sf.active = 1";
#    $sth = $dbh->prepare($sql);
#    $sth->execute($category);
#    while(my ($sf_id) = $sth->fetchrow_array){
#      print "RETIRING SUBFEATURE $sf_id.\n" if($VERBOSE);
#      change_subfeature_status($dbh, $sf_id, 0, $user_id);
#    }

    print "\nRetired $retired_p projects and $retired_i issues in category $category.\n\n";
  }
  elsif($category and $type and ($type eq "flags")){
    my $retired_f = 0;
    $sql = "SELECT id FROM ".$type.
           " WHERE seen = 0 AND flag_name = ? AND issue_id>-1 AND checked_date IS NULL";
    $sth = $dbh->prepare($sql);
    $sth->execute($category);
    while(my ($id) = $sth->fetchrow_array){
      print "RETIRING ".uc($type)." $id.\n" if($VERBOSE);
      if($WRITE){
	change_status($dbh, $id, "AutoResolved", $user_id, $type);
      }
      $retired_f++;
    }
    print "\nRetired $retired_f $type.\n\n";
  }
  else{
    die "\n*** Not implemented yet in analyse_seen_flags. ***\n";
  }

}


=head2 get_unseen_issues

  Arg [1]    : db connection
  Arg [2]    : issue category-id
  Description: retrieve all issues with a "seen" flag of 0
  Returntype : array-ref

=cut


sub get_unseen_issues {
    my ($dbh, $category_id, $higher_priority) = @_;

    my @ids;
    my $sth;

    my $sql = "SELECT DISTINCT i.id FROM projects p, issues i, issue_statuses iss ".
              "WHERE p.id = i.project_id AND i.seen = 0 ".
	      "AND i.status_id=iss.id ".
	      "AND iss.name != \"".$REJECTED_STATUS."\"";

    if($category_id){
      $sql .= " AND i.category_id = ?";
      $sth = $dbh->prepare($sql);
      $sth->execute($category_id);
    }
    elsif($higher_priority){
      $sql .= " AND i.priority_id > ?";
      $sth = $dbh->prepare($sql);
      $sth->execute($higher_priority);
    }
    else{
      $sth = $dbh->prepare($sql);
      $sth->execute();
    }

    while(my ($i_id) = $sth->fetchrow_array){
      push(@ids, $i_id);
    }

    return \@ids;
}


=head2 get_flag_types

  Arg [1]    : db connection
  Arg [2]    : [boolean; get only active/non-active flags]
  Description: get name and count of all types of flags found in db
  Returntype : none

=cut

sub get_flag_types {
  my ($dbh, $active) = @_;

  my %flags;
  my $sql = "SELECT flag_name, count(*) FROM flags ".
            "WHERE issue_id>0 ";
  if(defined $active){
    if($active == 1){
      $sql .= "AND checked_date IS NULL ";
    }
    else{
      $sql .= "AND checked_date IS NOT NULL ";
    }
  }
  $sql .= "GROUP BY flag_name;";
  my $sth = $dbh->prepare($sql);
  $sth->execute();
  while(my ($flagname, $flagcount) = $sth->fetchrow_array){
    $flags{$flagname} = $flagcount;
  }

  return \%flags;
}


=head2 get_status

  Arg [1]    : db connection
  Arg [2]    : project/issue-id
  Arg [3]    : type of element
  Description: get the status of a project
  Returntype : none

=cut

sub get_status {
  my ($dbh, $id, $type) = @_;

  my $table;
  if($type eq "gene" or $type eq "project"){
    $table = "projects";
  }elsif($type eq "transcript" or $type eq "issue"){
    $table = "issues";
  }
  my $sql = "SELECT iss.name FROM ".$table." p , issue_statuses iss ".
            "WHERE iss.id=p.status_id AND p.id=?";
  my $sth = $dbh->prepare($sql);
  $sth->execute($id);
  my ($status) = $sth->fetchrow_array();

  return $status;
}


=head2 change_status

  Arg [1]    : db connection
  Arg [2]    : project/issue-id
  Arg [3]    : name of new status
  Arg [4]    : type of element
  Description: update the status of a project
  Returntype : none

=cut

sub change_status {
  my ($dbh, $id, $new_status, $user_id, $type) = @_;

  if(!exists $STATUS{$new_status}){
    die "UNKNOWN STATUS $new_status.\n";
  }
  my $status_id = $STATUS{$new_status};
  #print "SETTING ".uc($type)." $id TO $new_status.\n" if($VERBOSE);
  my ($table, $sql, $sth, $changed);

  if($WRITE){
    if(($type eq "gene") or ($type eq "project")){
      $sql = "UPDATE projects p, issue_statuses iss ".
             "SET p.status_id=iss.id WHERE iss.name=? AND p.id=?";
      $sth = $dbh->prepare($sql);
      $changed = $sth->execute($new_status, $id);
    }elsif(($type eq "transcript") or ($type eq "issue")){
      $sql = "UPDATE issues i, issue_statuses iss ".
	     "SET i.status_id=iss.id WHERE iss.name=? AND i.id=?";
      $sth = $dbh->prepare($sql);
      $changed = $sth->execute($new_status, $id);
    }else{
      if($type eq "flags"){
	$sql = "UPDATE flags SET checked_date=?, reason=? WHERE id=?";
	$sth = $dbh->prepare($sql);
	my ($day, $month, $year) = (localtime)[3,4,5];
	my $resolved = sprintf("%04d-%02d-%02d", $year+1900, $month+1, $day);
	$sth->execute($resolved, $new_status, $id);
      }else{
	die "Can't update type $type.\n";
      }
    }

    if($changed){
      #write history
      my $notes = "Changed ".$type." ".$id." to status $new_status.";
      write_history($dbh, $id, $type, $user_id, $notes);
    }
  }
}


=head2 change_subfeature_status

  Arg [1]    : db connection
  Arg [2]    : subfeature-id
  Arg [3]    : name of new stutus
  Description: update the status of a subfeature (retire or revive)
  Returntype : none

=cut

sub change_subfeature_status {
  my ($dbh, $subfeature_id, $new_status, $user_id) = @_;

  #print "SETTING SUBFEATURE $subfeature_id TO $new_status.\n" if($VERBOSE);

  if($WRITE){
    #update status
    my $sql = 'UPDATE subfeatures SET active=? WHERE id=?';
    my $sth = $dbh->prepare($sql);
    my $changed = $sth->execute($new_status, $subfeature_id);

    if($changed){
      #write history
      my $notes = "Retired subfeature $subfeature_id.";
      write_history($dbh, $subfeature_id, 'subfeature', $user_id, $notes);
    }
  }
}


=head2 store_tec

  Arg [1]    : db connection
  Arg [2]    : prepare hash
  Arg [3]    : tec object
  Description: store a TEC entry
  Returntype : db id

=cut

sub store_tec {
  my ($tracking_dbh, $prepare_hash, $element, $user_id) = @_;

  my $new_tec_id = 0;
  #does the identifier exists?
  my $check = name_is_present($tracking_dbh, $prepare_hash, "tec", $element);

  if( scalar @$check == 1){
    print "Found existing TEC entry for issue ".$element->{'issue_id'}.": ".$check->[0]."\n";
    return($check->[0]);
  }
  elsif( scalar @$check > 1){
    #should not happen: non-unique id
    print STDERR "\nmultiple vals returned for query: ".join(", ", @$check)."!\n";
    return(0);
  }
  else{
    #store new data
    print ">INSERTING TEC, ".$element->{'issue_id'}.", ".$element->{'issue_name'}.", ".$element->{'sel_remark'}."\n"
      if $VERBOSE;
    $prepare_hash->{'new_tec'}->execute($element->{'issue_id'}, $element->{'issue_version'},
					$element->{'issue_name'}, $element->{'flag_id'},
					$element->{'sel_remark'}, $user_id, $element->{'result'});
    $new_tec_id = $tracking_dbh->last_insert_id(undef, undef, undef, undef);
  }

  return $new_tec_id;
}


=head2 store_tec_feature

  Arg [1]    : db connection
  Arg [2]    : prepare hash
  Arg [3]    : tec feature object
  Description: store a sub-part of a TEC entry
  Returntype : db id

=cut

sub store_tec_feature {
  my ($tracking_dbh, $prepare_hash, $tecf, $user_id) = @_;

  my $new_tecf_id = 0;
  #does the identifier exists?
  my $check = name_is_present($tracking_dbh, $prepare_hash, "tec_feature", $tecf);

  if( scalar @$check == 1){
    print "Found existing TEC-Feature entry for issue ".$tecf->{'tec_id'}."\n";
    return(0);
  }
  elsif( scalar @$check > 1){
    #should not happen: non-unique id
    print STDERR "\nmultiple vals returned for ".$tecf->{'tec_id'}." query: ".join(", ", @$check)."!\n";
    return(0);
  }
  else{
    #store new data
    print ">INSERTING TECFEATURE, ".$tecf->{'tec_id'}.", ".$tecf->{'type'}.", ".$tecf->{'start'}."-".$tecf->{'end'}."\n"
      if $VERBOSE;
    $new_tecf_id = $prepare_hash->{'new_tecfeature'}->execute($tecf->{'tec_id'}, $tecf->{'type'}, $tecf->{'chrom'},
							      $tecf->{'start'}, $tecf->{'end'}, $tecf->{'strand'},
							      $tecf->{'sequence'});
  }

  return $new_tecf_id;
}


=head2 get_custom_value

  Arg [1]    : db connection
  Arg [2]    : custom_type
  Arg [3]    : issue_id
  Description: get custom value stored for given issue
  Returntype : custom value

=cut

sub get_custom_value {
  my ($prepare_hash, $custom_type, $issue_id) = @_;

  my $custom_val;
  my $sth = $prepare_hash->{'custom_value'};
  $sth->execute($issue_id, $custom_type);
  ($custom_val) = $sth->fetchrow_array;

  return $custom_val;
}


=head2 check_tmp_table

  Arg [1]    : db connection
  Arg [2]    : type of value
  Arg [3]    : name of value
  Description: check tmp table for given value, insert if not exists
  Returntype : field_value in tmp value

=cut

sub check_tmp_table{
  my ($dbh, $field_group, $field_name) = @_;

  my $tmp_table = "tmp_values";

  my $sql = "SELECT field_value FROM ".$tmp_table.
    " WHERE field_group = ? AND field_name = ?";

  my $dbh_flag = $dbh->prepare($sql) or die $dbh->errstr;
  $dbh_flag->execute($field_group, $field_name);
  my ($field_value) = $dbh_flag->fetchrow_array;
  if(!$field_value){
    #doesnt exist, create
    $sql = "INSERT INTO ".$tmp_table.
      " SET field_group = ?, field_value = 0, field_name = ?";
    $dbh->prepare($sql) or die $dbh->errstr;
    $dbh_flag->execute($field_group, $field_name) or die "Cant create new tmp-val entry for flag $field_name.\n";
  }

}


=head2 set_flag

  Arg [1]    : db connection
  Arg [2]    : issue-id
  Arg [3]    : name of flag
  Arg [4]    : created date
  Arg [5]    : issue-id causing the flag
  Arg [6]    : optional note
  Description: create a new flag indicating some problem with a transcript
  Returntype : db id of new flag

=cut

sub set_flag {
  my ($dbh, $issue_id, $flag_name, $created, $user_id, $cause_id, $note) = @_;

  my $dbID;
  if($WRITE){
    if(!$created){
      my ($day, $month, $year) = (localtime)[3,4,5];
      $created = sprintf("%04d-%02d-%02d", $year+1900, $month+1, $day);
    }
    if(!$note){
      $note = "";
    }

    #if this is a new flag type, also store in tmp table
    check_tmp_table($dbh, 'flag', $flag_name);

    my $sql = "INSERT INTO flags set issue_id = ?, flag_name = ?, flag_issue_id= ?, ".
      "created_date = ?, note = ?, seen = ?";

    my $dbh_flag = $dbh->prepare($sql) or die $dbh->errstr;
    $dbh_flag->execute($issue_id, $flag_name, $cause_id, $created, $note,  '1');
    $dbID = $dbh->last_insert_id(undef, undef, undef, undef);

    my $notes = "Created flag \"".$flag_name."\"";
    write_history($dbh, $issue_id, 'transcript', $user_id, $notes);
  }

  return $dbID;
}


=head2 get_flag

  Arg [1]    : db connection
  Arg [2]    : issue-id
  Arg [3]    : name of flag
  Arg [4]    : created date
  Arg [5]    : string is_resolved? (yes/no)
  Arg [6]    : boolean, is Arg[3] ref to list of names?
  Description: fetch a flag
  Returntype : array ref with hash refs

=cut

sub get_flag {
  my ($dbh, $issue_id, $flag_name, $created, $created_id, $resolved, $multiple_flags) = @_;

# TODO: use ? in query!

  my $sql = "SELECT id, issue_id, flag_name, flag_issue_id, created_date, checked_date, note, reason FROM flags ".
            "WHERE issue_id > 0 ";
  my (@queryargs, @flags);

  if($issue_id and ($issue_id =~ /^\d+$/)){
    $sql .= "AND issue_id = $issue_id ";
    #push(@queryargs, $issue_id);
  }
  if($flag_name){
    if(!$multiple_flags){
      if($flag_name =~ /^[\w\_\-]+$/){
        $sql .= "AND flag_name = '".$flag_name."' ";
        #push(@queryargs, $flag_name);
      }
      else{
	warn "Invalid flag name $flag_name.\n";
      }
    }
    else{
      print "GETTING MULTIPLE FLAGS: ".join(", ", @$flag_name)."\n" if $VERBOSE;
      my @sql_add = ();
      foreach my $m_flag_name (@$flag_name){
        if($m_flag_name =~ /^[\w\_\-]+$/){
	  push(@sql_add, " flag_name like \"%".$m_flag_name."%\" ");
	  #push(@queryargs, $m_flag_name);
        }
      }
      $sql .= "AND ( ".(join(" AND ", @sql_add)).") ";
    }
  }
  if($created_id and ($created_id =~ /^\d+$/)){
    $sql .= "AND flag_issue_id = $created_id ";
    #push(@queryargs, $created_id);
  }
  if($resolved){
    if($resolved =~ /no/i){
      $sql .= "AND checked_date IS NULL ";
    }
    elsif($resolved =~ /yes/i){
      $sql .= "AND checked_date IS NOT NULL ";
    }
    else{
      die "Don't understand flag request for resolved=$resolved.\n";
    }
  }
  if($created){
#needs safety-check
#    $sql .= "AND created = ? ";
#    push(@queryargs, $created);
  }

  my $dbh_flag = $dbh->prepare($sql) or die $dbh->errstr;
  $dbh_flag->execute() or die $dbh->errstr; # :-(

  while(my ($fid, $issueid, $flagname, $flag_issue_id, $created_date, $checked_date, $note, $reason) = $dbh_flag->fetchrow_array){
    my %flag = (
		'id'            => $fid,
		'issue_id'      => $issueid,
		'flag_name'     => $flagname,
		'flag_issue_id' => $flag_issue_id,
		'created_date'  => $created_date,
		'checked_date'  => $checked_date,
		'note'          => $note,
		'reason'        => $reason,
	       );
    push(@flags, \%flag);
  }

  return \@flags;
}


=head2 resolve_flag

  Arg [1]    : db connection
  Arg [2]    : issue-id
  Arg [3]    : name of flag
  Arg [4]    : resolved date
  Description: resolve a flag by setting its checked-date
  Returntype : none

=cut

sub resolve_flag {
  my ($dbh, $issue_id, $flag_name, $checked) = @_;

  if($WRITE){
    my $sql = "UPDATE flags SET checked_date = ? WHERE issue_id = ? and flag_name = ?";
    my $dbh_flag = $dbh->prepare($sql) or die $dbh->errstr;
    $dbh_flag->execute($checked, $issue_id, $flag_name);
  }
}


=head2 set_issue_flags

  Arg [1]    : db connection
  Arg [2]    : issue-id
  Arg [3]    : array ref with name(s) of flag(s)
  Arg [4]    : (optional) db prepared handle
  Description: add the name of flags to each issue
  Returntype : none

=cut


sub set_issue_flags {
  my ($dbh, $flag_issue_id, $flag_name_array, $prepare) = @_;

  if($WRITE){
    if(!$prepare){
      my $sql = "UPDATE issues SET active_flags = ? WHERE id = ?";
      $prepare = $dbh->prepare($sql) or die $dbh->errstr;
    }
    my $flag_names = join(", ", @$flag_name_array);
    my $ud = $prepare->execute($flag_names, $flag_issue_id)
      or die "Cant execute active_flag update.\n";
  }
}


=head2 clear_active_flags

  Arg [1]    : db connection
  Arg [2]    : look at issues with seen=0 only
  Description: re-set the active_flag field of all issues
  Returntype : none

=cut


sub clear_active_flags {
  my ($dbh, $use_unseen) = @_;

  my $sql = "UPDATE issues SET active_flags = NULL";
  if($use_unseen){
    $sql .= " WHERE seen = 0";
  }
  my $prepare = $dbh->prepare($sql) or die $dbh->errstr;
  my $res = $prepare->execute();
  print "Cleared $res active flags.\n";
}


# test func with direct sql

sub custom_prepare {
  my ($dbh, $sql) = @_;

  #die "\n$sql\n\n***********\nEMERGENCY BREAK\n************\n";

  my $dbsth = $dbh->prepare($sql) or die $dbh->errstr;
  return $dbsth;
}


=head2 write_history

  Arg [1]    : db handle
  Arg [2]    : id of element
  Arg [3]    : type of element
  Arg [4]    : user id
  Arg [5]    : opt. notes
  Arg [6]    : opt. old_value
  Arg [7]    : opt. new_value
  Arg [8]    : opt. changed_property
  Description: save modification to tracking-system history
  Returntype : none

=cut

sub write_history {
  my ($dbh, $element_id, $type, $user_id, $notes, $old_value, $new_value, $changed_property) = @_;

  return unless $WRITE;

  my $journalized_type;
  my $property = "attr";
  my $prop_key;

  my $sql = "INSERT INTO journals SET journalized_id = ?, journalized_type = ?, user_id = ?, ".
            "notes = ?, created_on = now();";
  my $dbh_make_history = $dbh->prepare($sql) or die $dbh->errstr;

  if($type eq "gene"){
    $journalized_type = "Project";
  }
  elsif($type eq "transcript"){
    $journalized_type = "Issue";
  }
  elsif($type eq "subfeature"){
    $journalized_type = "Subfeature";
  }
  elsif($type =~ /flag/){
    $journalized_type = "Flag";
  }  else{
    die "\n*** Can't make history with $type ($notes). ***\n";
  }

  #insert into history/journal
  my $changed = $dbh_make_history->execute( $element_id, $journalized_type, $user_id, $notes );
  my $dbID = $dbh->last_insert_id(undef, undef, undef, undef);

  #also add details?
  if($changed_property && (defined $old_value and defined $new_value) && !($type =~ /flag/)){

    if($changed_property =~ /priority/){
      $prop_key = "priority_id";
    }
    if($changed_property =~ /category/){
      $prop_key = "category_id";
    }

    $sql = "INSERT INTO journal_details SET journal_id = ?, property=?, prop_key=?, old_value=?, value=?;";
    my $dbh_history_detail = $dbh->prepare($sql) or die $dbh->errstr;

    $dbh_history_detail->execute( $dbID, $property, $prop_key, $old_value, $new_value);
  }
}


=head2 _check_strand

  Arg [1]    : strand
  Description: look at the strand and return valid value,
               private method
  Returntype : + / - / 0

=cut

sub _check_strand {
  my $strand = shift;

  my $usestrand = '0';

  if(!$strand){
    print "MISSING STRAND VALUE\n";
  }
  elsif(($strand eq '+') or ($strand eq '+1') or ($strand eq '1')){
    $usestrand = '+';
  }
  elsif(($strand eq '-') or ($strand eq '-1')){
    $usestrand = '-';
  }
  elsif(($strand eq '0') or ($strand eq '.')){
    $usestrand = '0';
  }
  else{
    print "UNSUPPORTED STRAND VALUE: ".$strand."\n";
  }

  return $usestrand;
}


=head2 _check_id

  Arg [1]    : id
  Description: replace unwanted characters from the id with "_",
               unwanted: . " ' \ ? :
  Returntype : valid id

=cut

sub _check_id {
  my $id = shift;

  if(!$id){
    print "MISSING ID!\n";
    return 0;
  }
#  $id =~ s/\./_/g;
#  $id =~ s/\"/_/g;
#  $id =~ s/\'/_/g;
#  $id =~ s/\\/_/g;
#  $id =~ s/\//_/g;
#  $id =~ s/\?/_/g;
#  $id =~ s/\:/_/g;

  #less clear, more efficient?
  $id =~ s/\.|\"|\'|\\|\/\?\:/_/g;

  return $id;
}


=head2 _check_date

  Arg [1]    : date
  Description: check date for correct format
  Returntype : valid date

=cut

sub _check_date {
  my($date, $use_now) = @_;

  my $checked_date;
  my ($day, $month, $year, $hour, $min, $sec);

  #use today's date?
  if(!$date && $use_now){
    ($day, $month, $year, $hour, $min, $sec) = (localtime)[3,4,5,2,1,0];
    $checked_date = sprintf("%04d-%02d-%02d %02d-%02d-%02d", $year+1900, $month+1, $day, $hour, $min, $sec);
  }
  elsif($date){
    #check format
    my ($odate, $otime) = split('T', $date);
    ($year, $month, $day) = split('-', $odate);
    $otime =~ /^([\d\:]+)[Z|\+]/;
    $otime = $1;
    #TODO: TIMEZONE!
    #TODO: check for valid content!
    if(!$otime){ die "INVALID TIME FORMAT: $date.\n$year, $month, $day.\n$otime.\n"; }
    ($hour, $min, $sec) = split(':', $otime);
    $checked_date = sprintf("%04d-%02d-%02d %02d-%02d-%02d", $year, $month, $day, $hour, $min, $sec);
  }else{
    print "MISSING DATE\n";
    return 0;
  }

  return $checked_date;
}


=head2 get_issues

  Arg [1]    : db handle
  Arg [2]    : chromosome
  Arg [3]    : start
  Arg [4]    : end
  Arg [5]    : strand
  Arg [6]    : category id
  Arg [7]    : status id
  Description: fetch issues by locus (chromosome, start, end and strand) and category or by status and category.
               It can be accessed like this:
                 foreach my $issue (@$issues){
                   my %data = ();
                   for(my $i=0; $i< scalar @fields; $i++){
                     print "1\t".$fields[$i].": ".$issues->[0]->[$i]."\n";
                     $data{$fields[$i]} = $issues->[0]->[$i];
                   }
                   push(@results, \%data);
                 }
  Returntype : arrayref with results, arrayref with field names

=cut

sub get_issues {
  my ($tracking_dbh, $chrom, $start, $end, $strand, $category, $statuses, $prepared) = @_;

  my @results;
  my @dbfields = qw(id project_id subject description status_id priority_id created_on updated_on 
		    Tchrom Tstart Tend Tstrand);
  my @fields   = qw(transcript_id gene_id name description status_id priority_id created_on 
		    updated_on chrom start end strand);
  my ($sth, $sql);

  if($prepared){
    $prepared->execute($chrom, $end, $start, $strand, $category);
  }
  elsif (defined $chrom && defined $start && defined $end && defined $strand){
    $sql = "SELECT ".join(", ", @dbfields)." FROM issues WHERE Tchrom = ? AND Tstart <= ? ".
           "AND Tend >= ? AND Tstrand = ? and category_id = ?";

    $sth = $tracking_dbh->prepare($sql);
    $sth->execute($chrom, $end, $start, $strand, $category);
  }
  elsif(defined $category && defined $statuses){
    $sql = "SELECT ".join(", ", @dbfields)." FROM issues WHERE category_id = ? AND status_id IN(".$statuses.");";
    #print ">>$sql\n\n";
    $sth = $tracking_dbh->prepare($sql);
    $sth->execute($category);
  }
  else{
    print "$start, $end, $strand, $category, $statuses\n";
    die "missing locus or status.\n";
  }

  while(my $issues = $sth->fetchrow_arrayref()) {
    my %data = ();
    #print "ISSUE:\n";
    for(my $i=0; $i< scalar @fields; $i++){
      #print "1\t".$fields[$i].": ".$issues->[$i]."\n";
      $data{$fields[$i]} = $issues->[$i];
    }
    push(@results, \%data);
  }
  $sth->finish;

  return(\@results, \@fields);
}


=head2 get_subfeatures

  Arg [1]    : db handle
  Arg [2]    : chromosome
  Arg [3]    : start
  Arg [4]    : end
  Arg [5]    : strand
  Arg [6]    : category id
  Arg [7]    : status id
  Arg [8]    : (type of subfeature)
  Description: fetch subfeatures by locus (chromosome, start, end and strand) and category
               or by status and category.
               It can be accessed like this:
                 foreach my $issue (@$issues){
                   my %data = ();
                   for(my $i=0; $i< scalar @fields; $i++){
                     print "1\t".$fields[$i].": ".$issues->[0]->[$i]."\n";
                     $data{$fields[$i]} = $issues->[0]->[$i];
                   }
                   push(@results, \%data);
                 }
  Returntype : arrayref with results, arrayref with field names

=cut

sub get_subfeatures {
  my ($tracking_dbh, $chrom, $start, $end, $strand, $category, $statuses, $type) = @_;

  my @results;
  my @dbfields = qw(sf.id i.id i.subject sf.subfeature_chr sf.subfeature_start sf.subfeature_end 
		    sf.subfeature_strand sf.subfeature_type);
  my @fields   = qw(feature_id transcript_id transcript_name chrom start end strand type);
  my ($sth, $sql);

  #TODO: Add status check!

  if (defined $chrom && defined $start && defined $end && defined $strand){
    $sql = "SELECT ".join(", ", @dbfields)." FROM subfeatures sf, issues i WHERE sf.subfeature_chr = ? ".
           "AND sf.subfeature_start <= ? AND sf.subfeature_end >= ? AND sf.subfeature_strand = ? ".
           "AND i.id=sf.issue_id AND i.category_id = ?";
    if(defined $type){
      $sql .= " AND sf.subfeature_type = ?";
      $sth = $tracking_dbh->prepare($sql);
      $sth->execute($chrom, $end, $start, $strand, $category, $type);
    }
    else{
      $sth = $tracking_dbh->prepare($sql);
      $sth->execute($chrom, $end, $start, $strand, $category);
    }
  }
  else{
    print "$start, $end, $strand, $category, $statuses\n";
    die "missing info.\n";
  }

  while(my $issues = $sth->fetchrow_arrayref()) {
    my %data = ();
    for(my $i=0; $i< scalar @fields; $i++){
      #print "1\t".$fields[$i].": ".$issues->[$i]."\n";
      $data{$fields[$i]} = $issues->[$i];
    }
    push(@results, \%data);
  }
  $sth->finish;

  return(\@results, \@fields);
}


=head2 get_data_by_id

  Arg [1]    : db handle
  Arg [2]    : type of element
  Arg [3]    : id of element
  Description: fetch specific data (gene/transcript/subfeature) from db
  Returntype : ref to object hash

=cut

sub get_data_by_id {
  my ($tracking_dbh, $type, $id) = @_;

  my ($sth, $sql);
  my $c = 0;
  my %object = ();
  my ($dbid, $name, $description, $created_on, $updated_on, $chrom, 
      $start, $end, $strand, $category, $status, $project_id, $subject);

  if($type eq "gene"){
    $sql   = "SELECT id, name, description, created_on, updated_on, Gchrom, Gstart, Gend, Gstrand ".
             "from projects where id=?";
    $sth   = $tracking_dbh->prepare($sql);
    $sth->execute( $id );
    while(($dbid, $name, $description, $created_on, $updated_on, $chrom, 
	   $start, $end, $strand) = $sth->fetchrow_array()){
      if($c++ > 1){ die "ID $id appears more than once!"; };
      %object = (
		 'id'          => $dbid,
		 'name'        => $name,
		 'description' => $description,
		 'created'     => $created_on,
		 'updated'     => $updated_on,
		 'chromosome'  => $chrom,
		 'start'       => $start,
		 'end'         => $end,
		 'strand'      => $strand,
		);
    }
  }
  elsif($type eq "transcript"){
    $sql   = "SELECT i.id, i.project_id, i.subject, i.description, i.created_on, i.updated_on, ".
             "i.Tchrom, i.Tstart, i.Tend, i.Tstrand, c.name, s.name ".
	     "FROM issues i, issue_categories c , issue_statuses s ".
	     "WHERE i.category_id=c.id AND i.status_id=s.id and i.id=?";

    $sth   = $tracking_dbh->prepare($sql);
    $sth->execute( $id );
    while(($dbid, $project_id, $subject, $description, $created_on, $updated_on, $chrom, 
	   $start, $end, $strand, $category, $status) = $sth->fetchrow_array()){
      if($c++ > 1){ die "ID $id appears more than once!"; };
      %object = (
		 'id'          => $dbid,
		 'gene_id'     => $project_id,
		 'name'        => $subject,
		 'description' => $description,
		 'created'     => $created_on,
		 'updated'     => $updated_on,
		 'chromosome'  => $chrom,
		 'start'       => $start,
		 'end'         => $end,
		 'strand'      => $strand,
		 'category'    => $category,
		 'status'      => $status,
		 'subfeatures' => [],
		);

    }
  }
  elsif($type eq "subfeature"){
	    $sql   = "SELECT s.id, s.subfeature_type, s.subfeature_chr,s.subfeature_start, ".
	             "s.subfeature_end, s.subfeature_strand ".
                     "FROM subfeatures s WHERE s.id=?;";
	    $sth   = $tracking_dbh->prepare($sql);
	    $sth->execute( $id );
	    while(($dbid, $type, $chrom, $start, $end, $strand, $category, $status)
		  = $sth->fetchrow_array){
	      %object = (
			 'id'          => $dbid,
			 'type'        => $type,
			 'chromosome'  => $chrom,
			 'start'       => $start,
			 'end'         => $end,
			 'strand'      => $strand,
			);
	    }
  }
  else{
    die "\nCan't get object of type $type.\n";
  }
  $sth->finish;

  return(\%object);
}


=head2 get_data_by_name

  Arg [1]    : db handle
  Arg [2]    : type of element
  Arg [3]    : id of element
  Description: fetch specific data (gene/transcript/subfeature) from db
  Returntype : ref to object hash

=cut

sub get_data_by_name {
  my ($tracking_dbh, $type, $name) = @_;

  my ($sth, $sql);
  my $c = 0;
  my %object = ();

  if($type eq "gene"){
    $sql   = "SELECT id from projects where name=?";
    $sth   = $tracking_dbh->prepare($sql);
    $sth->execute( $name );
    while(my ($dbidd) = $sth->fetchrow_array()){
      %object = (
		 'id'          => $dbidd,
		 'name'        => $name,
		);

      if($c++ > 1){ die "Name $name appears more than once!"; };
    }
  }
  elsif($type eq "transcript"){
    $sql   = "SELECT i.id, i.project_id, i.subject, i.description, i.created_on, i.updated_on, ".
             "i.Tchrom, i.Tstart, i.Tend, i.Tstrand, c.name, s.name ".
	     "FROM issues i, issue_categories c , issue_statuses s ".
	     "WHERE i.category_id=c.id AND i.status_id=s.id and i.subject=?";
    $sth   = $tracking_dbh->prepare($sql);
    $sth->execute( $name );
    while(my ($dbid, $project_id, $subject, $description, $created_on, $updated_on, $chrom, 
	   $start, $end, $strand, $category, $status) = $sth->fetchrow_array()){
      %object = (
		 'id'          => $dbid,
		 'gene_id'     => $project_id,
		 'name'        => $subject,
		 'description' => $description,
		 'created'     => $created_on,
		 'updated'     => $updated_on,
		 'chromosome'  => $chrom,
		 'start'       => $start,
		 'end'         => $end,
		 'strand'      => $strand,
		 'category'    => $category,
		 'status'      => $status,
		 'subfeatures' => [],
		);

      if($c++ > 1){ die "ID $name appears more than once!"; };
    }
  }
  elsif($type eq "subfeature"){
    die "NOT YET SUPPORTED.\n";
#	    $sql   = "SELECT s.id, s.subfeature_type, s.subfeature_chr,s.subfeature_start, ".
#	             "s.subfeature_end, s.subfeature_strand ".
#                    "FROM subfeatures s, subfeatures_features sf ".
#                    "WHERE s.id=sf.subfeature_id AND sf.subfeature_id=?;";
#	    $sth   = $tracking_dbh->prepare($sql);
#	    $sth->execute( $id );
#	    while(my ($dbid, $type, $chrom, $start, $end, $strand, $category, $status) = $sth->fetchrow_array){
#	      %object = (
#			 'id'          => $dbid,
#			 'type'        => $type,
#			 'chromosome'  => $chrom,
#			 'start'       => $start,
#			 'end'         => $end,
#			 'strand'      => $strand,
#			);
#	    }
  }
  else{
    die "\nCan't get object of type $type.\n";
  }
  $sth->finish;

  return(\%object);
}


=head2 get_genes

  Arg [1]    : db handle
  Arg [2]    : bool get-id-only?
  Arg [3]    : category id OR
  Arg [4]    : id of element OR
  Arg [5]    : id of transcript of project
  Arg [6]    : (optional) specific db field to fetch
               for option get-id-only
  Arg [7]    : (optional) name of chromosome to look at
  Description: fetch specific project / gene from db
  Returntype : ref to array of object hashes ref OR
               ref to array of project ids if get-id-only==1

=cut

sub get_genes {
  my ($tracking_dbh, $id_only, $category_id, $id, $transcript_id, $desired_field, $inchrom) = @_;

  my ($dbid, $category, $subject, $description, $created_on, $updated_on, $status, 
      $chrom, $start, $end, $strand, $object);

  #look at specific chromosome only
  my $add_sql = "";
  if($inchrom){
    $inchrom =~ s/\.\&\*\%\#\?\"\'\;\=\(\)//g;
    $add_sql = " and i.Tchrom = '".$inchrom."'";
  }

  my ($sth, $sql);
  my $c = 0;
  my @objects;
  if(!$desired_field){
    $desired_field = "id";
  }

  if($id_only){
    if($id){
      $sql   = "SELECT p.".$desired_field." from projects p where p.id=?".$add_sql;
      $sth   = $tracking_dbh->prepare($sql);
      $sth->execute( $id );
      while(my ($dbid1) = $sth->fetchrow_array()){
	push(@objects, $dbid1);
	if($c++ > 1){ die "ID $id appears more than once!"; };
      }
    }
    elsif($transcript_id){
      $sql   = "SELECT p.".$desired_field." from projects p, issues i ".
	       "where p.id=i.project_id and i.id=?".$add_sql;
      $sth   = $tracking_dbh->prepare($sql);
      $sth->execute( $transcript_id );
      while(my ($dbid2) = $sth->fetchrow_array()){
	push(@objects, $dbid2);
	if($c++ > 1){ die "ID $transcript_id appears more than once!"; };
      }
    }
    elsif($category_id){
      $sql   = "SELECT p.".$desired_field." from projects p, issues i ".
               "where p.id=i.project_id and i.category_id=?".$add_sql;
      $sth   = $tracking_dbh->prepare($sql);
      $sth->execute( $category_id );
      while(my ($dbid3) = $sth->fetchrow_array()){
	push(@objects, $dbid3);
      }
    }
  }
  elsif($id){
    $sql   = "SELECT p.id, i.category_id, p.name, p.description, p.created_on, p.updated_on, ".
             "p.status, p.Gchrom, p.Gstart, p.Gend, p.Gstrand from projects p, issues i ".
             "where p.id=i.project_id and p.id=?";
    $sth   = $tracking_dbh->prepare($sql);
    $sth->execute( $id );
    while(($dbid, $category, $subject, $description, $created_on, $updated_on, $status, 
	      $chrom, $start, $end, $strand) = $sth->fetchrow_array()){
      $object = create_gene_object($tracking_dbh, $dbid, $category, $subject, $description, $created_on, 
				      $updated_on, $status, $chrom, $start, $end, $strand);
      push(@objects, $object);
      if($c++ > 1){ die "ID $transcript_id appears more than once!"; };
    }
  }
  elsif($transcript_id){
    $sql   = "SELECT p.id, i.category_id, p.name, p.description, p.created_on, p.updated_on, ".
             "p.status, p.Gchrom, p.Gstart, p.Gend, p.Gstrand from projects p, issues i ".
             "where p.id=i.project_id and i.id=?";
    $sth   = $tracking_dbh->prepare($sql);
    $sth->execute( $transcript_id );

    while(($dbid, $category, $subject, $description, $created_on, $updated_on, $status, 
	      $chrom, $start, $end, $strand) = $sth->fetchrow_array()){
      $object = create_gene_object($tracking_dbh, $dbid, $category, $subject, $description, $created_on, 
				      $updated_on, $status, $chrom, $start, $end, $strand);
      push(@objects, $object);
      if($c++ > 1){ die "ID $transcript_id appears more than once!"; };
    }
  }
  elsif($category_id){
    $sql   = "SELECT p.id, p.name, p.description, p.created_on, p.updated_on, ".
             "p.status, p.Gchrom, p.Gstart, p.Gend, p.Gstrand from projects p, issues i ".
             "where p.id=i.project_id and i.category_id=?";
    $sth   = $tracking_dbh->prepare($sql);
    $sth->execute( $category_id );

    while(($dbid, $subject, $description, $created_on, $updated_on, $status, 
	      $chrom, $start, $end, $strand) = $sth->fetchrow_array()){
      $object = create_gene_object($tracking_dbh, $dbid, $category_id, $subject, $description, $created_on, 
				      $updated_on, $status, $chrom, $start, $end, $strand);
      push(@objects, $object);
    }
  }
  $sth->finish;

  return(\@objects);
}


=head2 get_transcripts

  Arg [1]    : db handle
  Arg [2]    : bool get-id-only?
  Arg [3]    : category id OR
  Arg [4]    : id of element OR
  Arg [5]    : id of gene of transcript
  Arg [6]    : (optional) specific db field to fetch
               for option get-id-only
  Arg [7]    : (optional) name of chromosome to look at
  Description: fetch specific issues / transcript from db
  Returntype : ref to array of object hashes ref OR
               ref to array of project ids if get-id-only==1

=cut

sub get_transcripts {
  my ($tracking_dbh, $id_only, $category_id, $id, $gene_id, $desired_field, $chrom) = @_;

  #look at specific chromosome only
  my $add_sql = "";
  if($chrom){
    $add_sql = " and i.Tchrom = 22";
  }

  my ($sth, $sql, $dbid);
  my $c = 0;
  my @objects;
  if(!$desired_field){
    $desired_field = "id";
  }

  if($id_only){
    if($id){
      $sql   = "SELECT i.".$desired_field." from issues i where i.id=?".$add_sql;
      $sth   = $tracking_dbh->prepare($sql);
      $sth->execute( $id );
      while(($dbid) = $sth->fetchrow_array()){
	push(@objects, $dbid);
	if($c++ > 1){ die "ID $id appears more than once!"; };
      }
    }
    elsif($gene_id){
      $sql   = "SELECT i.".$desired_field." from projects p, issues i ".
	       "where p.id=i.project_id and p.id=?".$add_sql;
      $sth   = $tracking_dbh->prepare($sql);
      $sth->execute( $gene_id );
      while(($dbid) = $sth->fetchrow_array()){
	push(@objects, $dbid);
      }
    }
    elsif($category_id){
      $sql   = "SELECT i.".$desired_field." from issues i where i.category_id=?".$add_sql;
      $sth   = $tracking_dbh->prepare($sql);
      $sth->execute( $category_id );
      while(($dbid) = $sth->fetchrow_array()){
	push(@objects, $dbid);
      }
    }
  }
#  elsif($id){
#    $sql   = "SELECT p.id, i.category_id, p.name, p.description, p.created_on, p.updated_on, ".
#             "p.status, p.Gchrom, p.Gstart, p.Gend, p.Gstrand from projects p, issues i ".
#             "where p.id=i.project_id and p.id=?";
#    $sth   = $tracking_dbh->prepare($sql);
#    $sth->execute( $id );
#    while(my ($dbid, $category, $subject, $description, $created_on, $updated_on, $status, 
#	      $chrom, $start, $end, $strand) = $sth->fetchrow_array()){
#      my $object = create_gene_object($dbid, $category, $subject, $description, $created_on, 
#				      $updated_on, $status, $chrom, $start, $end, $strand);
#      push(@objects, $object);
#      if($c++ > 1){ die "ID $transcript_id appears more than once!"; };
#    }
#  }
#  elsif($gene_id){
#    $sql   = "SELECT p.id, i.category_id, p.name, p.description, p.created_on, p.updated_on, ".
#             "p.status, p.Gchrom, p.Gstart, p.Gend, p.Gstrand from projects p, issues i ".
#             "where p.id=i.project_id and i.id=?";
#    $sth   = $tracking_dbh->prepare($sql);
#    $sth->execute( $gene_id );

#    while(my ($dbid, $category, $subject, $description, $created_on, $updated_on, $status, 
#	      $chrom, $start, $end, $strand) = $sth->fetchrow_array()){
#      my $object = create_gene_object($dbid, $category, $subject, $description, $created_on, 
#				      $updated_on, $status, $chrom, $start, $end, $strand);
#      push(@objects, $object);
#      if($c++ > 1){ die "ID $gene_id appears more than once!"; };
#    }
#  }
#  elsif($category_id){
#    $sql   = "SELECT p.id, p.name, p.description, p.created_on, p.updated_on, ".
#             "p.status, p.Gchrom, p.Gstart, p.Gend, p.Gstrand from projects p, issues i ".
#             "where p.id=i.project_id and i.category_id=?";
#    $sth   = $tracking_dbh->prepare($sql);
#    $sth->execute( $category_id );

#    while(my ($dbid, $subject, $description, $created_on, $updated_on, $status, 
#	      $chrom, $start, $end, $strand) = $sth->fetchrow_array()){
#      my $object = create_gene_object($dbid, $category_id, $subject, $description, $created_on, 
#				      $updated_on, $status, $chrom, $start, $end, $strand);
#      push(@objects, $object);
#    }
#  }
  $sth->finish;

  return(\@objects);
}


=head2 issue_relation

  Arg [1]    : db handle
  Arg [2]    : transcript db id 1
  Arg [3]    : transcript db id 2
  Description: create a relation between two issues / transcripts
               useful to identify related problems quickly
  Returntype : none

=cut

sub issue_relation {
  my ($dbh, $dbid1, $dbid2) = @_;

  if($dbid1 > $dbid2){
    ($dbid1, $dbid2) = ($dbid2, $dbid1);
  }
  print "\tSetting relation between $dbid1 & $dbid2\n" if($VERBOSE);
  if($WRITE){
    my $relation_type = "relates";
    my $sql = "INSERT IGNORE INTO issue_relations SET issue_from_id=?, issue_to_id=?, relation_type=?;";
    my $sth = $dbh->prepare($sql);
    $sth->execute($dbid1, $dbid2, $relation_type);
  }
}


=head2 create_gene_object

  Arg [1]    : db handle
  Arg [2]    : gene db id
  Arg [3]    : category id
  Arg [4]    : name of gene
  Arg [5]    : description...
  Description: create gene object hash from database
  Returntype : ref to gene object hash

=cut

sub  create_gene_object {
  my ($dbh, $dbid, $category_id, $subject, $description, $created_on, 
      $updated_on, $status, $chrom, $start, $end, $strand) = @_;

  #TODO:
  #check values

  my $category = "";
  my $sth = $dbh->prepare( "SELECT name from issue_categories where id=?" );
  $sth->execute($category);
  ($category) = $sth->fetchrow_array();
  $sth->finish;

  my %object = (
		'id'          => $dbid,
		'name'        => $subject,
		'description' => $description,
		'created'     => $created_on,
		'updated'     => $updated_on,
		'chromosome'  => $chrom,
		'start'       => $start,
		'end'         => $end,
		'strand'      => $strand,
		'category'    => $category,
		'status'      => $status,
		'active'      => 1,
	       );

  return \%object;
}


=head2 set_date

  Arg [1]    : db handle
  Arg [2]    : prepared db handle
  Arg [3]    : object id
  Arg [4]    : object type
  Arg [5]    : new date
  Arg [6]    : user id
  Description: set/update the updated date for specific element
  Returntype : none

=cut

sub set_date {
  my ($tracking_dbh, $prepare_hash, $db_id, $type, $date, $user_id) = @_;

  if(!$date){
    my ($day, $month, $year, $hour, $min, $sec) = (localtime)[3,4,5,2,1,0];
    $date = sprintf("%04d-%02d-%02d %02d-%02d-%02d", $year+1900, $month+1, $day, $hour, $min, $sec);
  }
  #print "SETTING $type date to $date.\n" if($VERBOSE);
  #my $ind = 'set_'.$type.'_date';

  if($WRITE){
      $prepare_hash->{'set_'.$type.'_date'}->execute($date, $db_id)
	or die "ERROR IN DATE-UPDATE (".'set_'.$type.'_date'.").";

#      #write to history log automatically if changed
#      my $notes = "";
#      write_history($tracking_dbh, $transcript_id, "transcript", $user_id,
#		    $notes, $old_date, $priority_id, 'priority');
  }

  return 0;
}


=head2 set_issue_priority

  Arg [1]    : db handle
  Arg [2]    : transcript id
  Arg [3]    : new priority id
  Arg [4]    : user-id
  Description: update priority for specific transcript
  Returntype : none

=cut

sub set_issue_priority {
  my ($tracking_dbh, $prepare_hash, $transcript_id, $priority_id, $user_id) = @_;

  print "SETTING $transcript_id to $priority_id.\n" if($VERBOSE);

  if($WRITE){
    #set new priority
    $prepare_hash->{'set_priorities'}->execute($priority_id, $transcript_id)
      or die "ERROR IN PRIOR. UPDATE.";

#    #alternative version: only update if priority changed, also write to history
#    #get current priority
#    $prepare_hash->{'get_priority'}->execute($transcript_id) or die "ERROR IN PRIOR. QUERY.";
#    my ($old_priority_id) = $prepare_hash->{'get_priority'}->fetchrow_array();

#    if($old_priority_id != $priority_id){

#      #set new priority
#      $prepare_hash->{'set_priorities'}->execute($priority_id, $transcript_id)
#	or die "ERROR IN PRIOR. UPDATE.";

#      #write to history log automatically if changed
#      my $notes = "";
#      write_history($tracking_dbh, $transcript_id, "transcript", $user_id,
#		    $notes, $old_priority_id, $priority_id, 'priority');

#    }
#    else{
#      print "NOTHING CHANGED.\n" if($VERBOSE);
#    }
  }

  return 0;
}


=head2 get_core_annotation

  Arg [1]    : db handle
  Description: get name & id of the base / core annotation (eg "HAVANA")
  Returntype : name (string), id (int)

=cut

sub get_core_annotation {
  my ($tracking_dbh) = @_;

  my $sth = $tracking_dbh->prepare("SELECT s.value, ic.id FROM settings s, issue_categories ic ".
				   "where ic.name=s.value and s.name='core_annotation';")
    or die "Cant prepare get_core_annotation sql\n";
  $sth->execute() or die "Cant execute get_core_annotation sql\n";
  my ($cname, $cid) = $sth->fetchrow_array();

  return($cname, $cid);
}


=head2 delete_projects

  Arg [1]    : db handle
  Arg [2]    : category name
  Arg [3]    : project name
  Arg [4]    : issue name
  Arg [5]    : indicator to confirm deletion of ALL entries
  Description: complete remove data from the db, including linked entries,
               using either category, project or issue.
               !Not to be used in production!
  Returntype : none

=cut

sub delete_projects {
  my ($dbh, $category, $project, $issue, $all) = @_;

  if(!$WRITE){
    print "NO-WRITE. NO-DELETE.\n";
    return 1;
  }
  if($ENVI eq 'prod'){
    print "DELETION POINTED TO PRODUCTION SERVER!\n";
    #return 1;
  }

  my ($sql1, $input);

  if($category){
    $sql1  = "SELECT i.id, p.id FROM projects p, issues i, issue_categories ic ".
            "WHERE p.id=i.project_id AND ic.id=i.category_id AND ic.name=?";
    $input = $category;
  }
  elsif($project){
    $sql1  = "SELECT i.id, p.id FROM projects p, issues i WHERE p.id=i.project_id AND p.name=?";
    $input = $project;
  }
  elsif($issue){
    $sql1  = "SELECT i.id, '0' FROM issues i WHERE i.subject=?";
    $input = $issue;
  }
  elsif($all){
    print "Nothing specified. Deleting everything!\n";

    #comment out to really remove all
    die "\n***************\nEMERGENCY BREAK\n***************\n";

    $dbh->do("truncate wikis;");
    $dbh->do("truncate wiki_contents;");
    $dbh->do("truncate journals;");
    $dbh->do("truncate journal_details;");
    $dbh->do("truncate custom_values;");
    $dbh->do("truncate projects_trackers;");
    $dbh->do("truncate projects;");
    $dbh->do("truncate subfeatures;");
    $dbh->do("truncate issues;");
    $dbh->do("truncate flags;");
    print "KILLED!\n";

    return 0;
  }

  my $sth1  = $dbh->prepare($sql1);
  my $sql2  = "DELETE w, wc FROM wikis w, wiki_contents wc WHERE wc.id=w.id AND w.project_id IN ";
  my $sql3  = "DELETE j, jd FROM journals j, journal_details jd WHERE jd.journal_id=j.id AND ".
              "j.journalized_type='Project' AND j.journalized_id IN ";
  my $sql4  = "DELETE cv FROM custom_values cv WHERE cv.customized_type='Project' AND cv.customized_id IN ";
  my $sql5  = "DELETE pt FROM projects_trackers pt WHERE pt.project_id IN ";
  my $sql6  = "DELETE p FROM projects p WHERE p.id IN ";
  my $sql7  = "DELETE sf FROM subfeatures sf WHERE sf.issue_id IN ";
  my $sql8  = "DELETE j, jd  FROM journals j LEFT JOIN journal_details jd ON jd.journal_id=j.id ".
              "WHERE j.journalized_type='Issue' AND j.journalized_id IN ";
  my $sql9  = "DELETE cv FROM custom_values cv WHERE cv.customized_type='Issue' AND id IN ";
  my $sql10 = "DELETE i FROM issues i WHERE i.id IN ";
  my $sql11 = "DELETE f FROM flags f WHERE f.issue_id IN ";


  my (@project_ids, @issue_ids);
  my ($issue_id, $project_id);
  $sth1->execute($input);
  while(($issue_id, $project_id) = $sth1->fetchrow_array){
    print "DELETING PROJECT $project_id / ISSUE $issue_id\n";

    push(@project_ids, $project_id);
    push(@issue_ids, $issue_id);
  }

  #delete in blocks of 20 if possible
  my $i = 0;
  my $l = 19;
  my $step = 20;
  if((scalar @project_ids) < 20){
    $l    = 4;
    $step = 5;
  }

  while(my @slice1 = (@project_ids[$i..$l])){
    print ".";
    $project_id = join(", ", @slice1);

    $dbh->do($sql2."(".$project_id.")") or die "PROBLEM WITH: ".$sql2."(".$project_id.")\n";
    $dbh->do($sql3."(".$project_id.")") or die "PROBLEM WITH: ".$sql3."(".$project_id.")\n";
    $dbh->do($sql4."(".$project_id.")") or die "PROBLEM WITH: ".$sql4."(".$project_id.")\n";
    $dbh->do($sql5."(".$project_id.")") or die "PROBLEM WITH: ".$sql5."(".$project_id.")\n";
    $dbh->do($sql6."(".$project_id.")") or die "PROBLEM WITH: ".$sql6."(".$project_id.")\n";

    $i += $step;
    $l += $step;
    if($l >= scalar @project_ids){
      $l = (scalar @project_ids) -1;
    }

  }

  print "\n";
  $i = 0;
  $l = 19;
  if((scalar @issue_ids) < 20){
    $l    = 4;
    $step = 5;
  }
  while(my @slice2 = (@issue_ids[$i..$l])){
    print ".";
    $issue_id = join(", ", @slice2);

    $dbh->do($sql7."(".$issue_id.")")  or die "PROBLEM WITH: ".$sql7."(".$issue_id.")\n";
    $dbh->do($sql8."(".$issue_id.")")  or die "PROBLEM WITH: ".$sql8."(".$issue_id.")\n";
    $dbh->do($sql9."(".$issue_id.")")  or die "PROBLEM WITH: ".$sql9."(".$issue_id.")\n";
    $dbh->do($sql10."(".$issue_id.")") or die "PROBLEM WITH: ".$sql10."(".$issue_id.")\n";
    $dbh->do($sql11."(".$issue_id.")") or die "PROBLEM WITH: ".$sql11."(".$issue_id.")\n";

    $i += $step;
    $l += $step;
    if($l >= scalar @issue_ids){
      $l = (scalar @issue_ids) -1;
    }

  }

  print "\nDELETION DONE!\n";
}


=head2 send_mail

 Arg [1]    : message text
 Arg [2]    : to-address
 Arg [3]    : from-address
 Arg [4]    : subject
 Description: Send email message
 Returntype : boolean, 0 for success

=cut

sub send_mail {
  my ($text, $uemail, $myemail, $subj) = @_;

  my $msg = MIME::Lite->new(
			    From    => $myemail,
			    To      => $uemail,
			    Subject => $subj,
			    Data    => $text
			   );

  $msg->send || return 0;

  return 1;
}


1;


__END__

TODOs

-create new category
-create new flag category?
-create entries in tmp table

